"""
认证API
"""
from fastapi import APIRouter, HTTPException, Depends
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from pydantic import BaseModel

from ..core.config import settings
from ..core.security import create_access_token, verify_token

router = APIRouter()
security = HTTPBearer(auto_error=False)


class LoginRequest(BaseModel):
    """登录请求"""
    password: str


class LoginResponse(BaseModel):
    """登录响应"""
    access_token: str
    token_type: str = "bearer"


class UserInfo(BaseModel):
    """用户信息"""
    username: str = "admin"
    role: str = "admin"


async def get_current_user(
    credentials: HTTPAuthorizationCredentials = Depends(security)
) -> dict:
    """获取当前用户"""
    if not credentials:
        raise HTTPException(status_code=401, detail="未登录")
    
    payload = verify_token(credentials.credentials)
    if not payload:
        raise HTTPException(status_code=401, detail="Token无效或已过期")
    
    return payload


@router.post("/login", response_model=LoginResponse)
async def login(request: LoginRequest):
    """登录"""
    if request.password != settings.ADMIN_PASSWORD:
        raise HTTPException(status_code=401, detail="密码错误")
    
    access_token = create_access_token({"sub": "admin", "role": "admin"})
    return LoginResponse(access_token=access_token)


@router.get("/me", response_model=UserInfo)
async def get_me(current_user: dict = Depends(get_current_user)):
    """获取当前用户信息"""
    return UserInfo(
        username=current_user.get("sub", "admin"),
        role=current_user.get("role", "admin")
    )


@router.post("/logout")
async def logout():
    """登出"""
    return {"message": "已登出"}


class ChangePasswordRequest(BaseModel):
    """修改密码请求"""
    old_password: str
    new_password: str


@router.post("/change-password")
async def change_password(
    request: ChangePasswordRequest,
    current_user: dict = Depends(get_current_user)
):
    """修改密码"""
    from ..core.config import settings
    import os
    
    # 验证旧密码
    if request.old_password != settings.ADMIN_PASSWORD:
        raise HTTPException(status_code=400, detail="当前密码错误")
    
    # 验证新密码
    if len(request.new_password) < 6:
        raise HTTPException(status_code=400, detail="新密码至少6位")
    
    # 更新.env文件中的密码
    env_path = os.path.join(os.path.dirname(os.path.dirname(os.path.dirname(__file__))), '.env')
    
    # 读取或创建.env文件
    env_content = ""
    if os.path.exists(env_path):
        with open(env_path, 'r', encoding='utf-8') as f:
            env_content = f.read()
    
    # 更新密码
    if 'ADMIN_PASSWORD=' in env_content:
        import re
        env_content = re.sub(
            r'ADMIN_PASSWORD=.*',
            f'ADMIN_PASSWORD={request.new_password}',
            env_content
        )
    else:
        env_content += f'\nADMIN_PASSWORD={request.new_password}\n'
    
    # 写入文件
    with open(env_path, 'w', encoding='utf-8') as f:
        f.write(env_content)
    
    # 更新运行时配置
    settings.ADMIN_PASSWORD = request.new_password
    
    return {"message": "密码修改成功"}

