"""
客户管理API
"""
import secrets
from typing import Optional
from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy import select, func
from sqlalchemy.ext.asyncio import AsyncSession
from sqlalchemy.orm import selectinload

from ..core.database import get_db
from ..core.config import settings
from ..models import Customer, Node
from ..models.node import NodeStatus
from ..schemas.customer import (
    CustomerCreate, 
    CustomerUpdate, 
    CustomerResponse, 
    CustomerListResponse
)
from .auth import get_current_user

router = APIRouter()


def generate_subscribe_token() -> str:
    """生成订阅令牌"""
    return secrets.token_urlsafe(24)


@router.get("", response_model=CustomerListResponse)
async def list_customers(
    page: int = Query(1, ge=1),
    page_size: int = Query(20, ge=1, le=100),
    keyword: Optional[str] = None,
    customer_type: Optional[str] = None,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取客户列表"""
    query = select(Customer)
    count_query = select(func.count(Customer.id))
    
    # 过滤条件
    if keyword:
        query = query.where(Customer.name.contains(keyword))
        count_query = count_query.where(Customer.name.contains(keyword))
    
    if customer_type:
        query = query.where(Customer.customer_type == customer_type)
        count_query = count_query.where(Customer.customer_type == customer_type)
    
    # 总数
    total = await db.scalar(count_query)
    
    # 分页
    query = query.order_by(Customer.created_at.desc())
    query = query.offset((page - 1) * page_size).limit(page_size)
    
    result = await db.execute(query)
    customers = result.scalars().all()
    
    # 构建响应
    items = []
    for customer in customers:
        # 统计节点数（运行中）
        node_count = await db.scalar(
            select(func.count(Node.id)).where(
                Node.customer_id == customer.id,
                Node.status == NodeStatus.RUNNING
            )
        )
        # 构建订阅链接
        subscribe_url = None
        if customer.subscribe_token and settings.SUBSCRIBE_BASE_URL:
            subscribe_url = f"{settings.SUBSCRIBE_BASE_URL}/api/subscribe/c/{customer.subscribe_token}"
        
        items.append(CustomerResponse(
            id=customer.id,
            name=customer.name,
            contact=customer.contact,
            customer_type=customer.customer_type,
            expire_at=customer.expire_at,
            remark=customer.remark,
            created_at=customer.created_at,
            updated_at=customer.updated_at,
            node_count=node_count or 0,
            subscribe_token=customer.subscribe_token,
            subscribe_url=subscribe_url
        ))
    
    return CustomerListResponse(items=items, total=total or 0)


@router.post("", response_model=CustomerResponse)
async def create_customer(
    data: CustomerCreate,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """创建客户"""
    customer = Customer(**data.model_dump())
    # 自动生成订阅令牌
    customer.subscribe_token = generate_subscribe_token()
    db.add(customer)
    await db.commit()
    await db.refresh(customer)
    
    # 构建订阅链接
    subscribe_url = None
    if customer.subscribe_token and settings.SUBSCRIBE_BASE_URL:
        subscribe_url = f"{settings.SUBSCRIBE_BASE_URL}/api/subscribe/c/{customer.subscribe_token}"
    
    return CustomerResponse(
        id=customer.id,
        name=customer.name,
        contact=customer.contact,
        customer_type=customer.customer_type,
        expire_at=customer.expire_at,
        remark=customer.remark,
        created_at=customer.created_at,
        updated_at=customer.updated_at,
        node_count=0,
        subscribe_token=customer.subscribe_token,
        subscribe_url=subscribe_url
    )


@router.get("/{customer_id}", response_model=CustomerResponse)
async def get_customer(
    customer_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取客户详情"""
    customer = await db.get(Customer, customer_id)
    if not customer:
        raise HTTPException(status_code=404, detail="客户不存在")
    
    node_count = await db.scalar(
        select(func.count(Node.id)).where(
            Node.customer_id == customer.id,
            Node.status == NodeStatus.RUNNING
        )
    )
    
    # 构建订阅链接
    subscribe_url = None
    if customer.subscribe_token and settings.SUBSCRIBE_BASE_URL:
        subscribe_url = f"{settings.SUBSCRIBE_BASE_URL}/api/subscribe/c/{customer.subscribe_token}"
    
    return CustomerResponse(
        id=customer.id,
        name=customer.name,
        contact=customer.contact,
        customer_type=customer.customer_type,
        expire_at=customer.expire_at,
        remark=customer.remark,
        created_at=customer.created_at,
        updated_at=customer.updated_at,
        node_count=node_count or 0,
        subscribe_token=customer.subscribe_token,
        subscribe_url=subscribe_url
    )


@router.put("/{customer_id}", response_model=CustomerResponse)
async def update_customer(
    customer_id: int,
    data: CustomerUpdate,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """更新客户"""
    customer = await db.get(Customer, customer_id)
    if not customer:
        raise HTTPException(status_code=404, detail="客户不存在")
    
    update_data = data.model_dump(exclude_unset=True)
    for key, value in update_data.items():
        setattr(customer, key, value)
    
    await db.commit()
    await db.refresh(customer)
    
    node_count = await db.scalar(
        select(func.count(Node.id)).where(
            Node.customer_id == customer.id,
            Node.status == NodeStatus.RUNNING
        )
    )
    
    # 构建订阅链接
    subscribe_url = None
    if customer.subscribe_token and settings.SUBSCRIBE_BASE_URL:
        subscribe_url = f"{settings.SUBSCRIBE_BASE_URL}/api/subscribe/c/{customer.subscribe_token}"
    
    return CustomerResponse(
        id=customer.id,
        name=customer.name,
        contact=customer.contact,
        customer_type=customer.customer_type,
        expire_at=customer.expire_at,
        remark=customer.remark,
        created_at=customer.created_at,
        updated_at=customer.updated_at,
        node_count=node_count or 0,
        subscribe_token=customer.subscribe_token,
        subscribe_url=subscribe_url
    )


@router.delete("/{customer_id}")
async def delete_customer(
    customer_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """删除客户"""
    customer = await db.get(Customer, customer_id)
    if not customer:
        raise HTTPException(status_code=404, detail="客户不存在")
    
    await db.delete(customer)
    await db.commit()
    
    return {"message": "删除成功"}


@router.post("/{customer_id}/refresh-token")
async def refresh_subscribe_token(
    customer_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """
    刷新客户订阅令牌
    
    生成新的订阅链接，旧链接将失效
    """
    customer = await db.get(Customer, customer_id)
    if not customer:
        raise HTTPException(status_code=404, detail="客户不存在")
    
    # 生成新令牌
    customer.subscribe_token = generate_subscribe_token()
    await db.commit()
    await db.refresh(customer)
    
    # 构建订阅链接
    subscribe_url = None
    if settings.SUBSCRIBE_BASE_URL:
        subscribe_url = f"{settings.SUBSCRIBE_BASE_URL}/api/subscribe/c/{customer.subscribe_token}"
    
    return {
        "message": "订阅令牌已刷新",
        "subscribe_token": customer.subscribe_token,
        "subscribe_url": subscribe_url
    }


@router.get("/{customer_id}/subscribe-info")
async def get_subscribe_info(
    customer_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """
    获取客户订阅信息
    
    返回订阅链接和节点统计
    """
    customer = await db.get(Customer, customer_id)
    if not customer:
        raise HTTPException(status_code=404, detail="客户不存在")
    
    # 如果没有订阅令牌，自动生成
    if not customer.subscribe_token:
        customer.subscribe_token = generate_subscribe_token()
        await db.commit()
        await db.refresh(customer)
    
    # 获取该客户的所有运行中节点
    result = await db.execute(
        select(Node).where(
            Node.customer_id == customer_id,
            Node.status == NodeStatus.RUNNING
        )
    )
    nodes = result.scalars().all()
    
    # 统计信息
    total_nodes = len(nodes)
    total_traffic_up = sum(n.traffic_up or 0 for n in nodes)
    total_traffic_down = sum(n.traffic_down or 0 for n in nodes)
    
    # 构建订阅链接
    subscribe_url = None
    if settings.SUBSCRIBE_BASE_URL:
        subscribe_url = f"{settings.SUBSCRIBE_BASE_URL}/api/subscribe/c/{customer.subscribe_token}"
    else:
        # 如果没有配置基础URL，使用相对路径提示
        subscribe_url = f"/api/subscribe/c/{customer.subscribe_token}"
    
    return {
        "customer_name": customer.name,
        "subscribe_token": customer.subscribe_token,
        "subscribe_url": subscribe_url,
        "node_count": total_nodes,
        "total_traffic_up": total_traffic_up,
        "total_traffic_down": total_traffic_down,
        "nodes": [
            {
                "id": n.id,
                "name": n.name,
                "status": n.status.value,
                "traffic_up": n.traffic_up,
                "traffic_down": n.traffic_down
            }
            for n in nodes
        ]
    }




