"""
仪表盘API
"""
from datetime import datetime, timedelta
from typing import List
from fastapi import APIRouter, Depends
from pydantic import BaseModel
from sqlalchemy import select, func
from sqlalchemy.ext.asyncio import AsyncSession

from ..core.database import get_db
from ..models import Customer, Server, Socks5Proxy, Node
from ..models.server import ServerStatus
from ..models.socks5 import Socks5Status
from ..models.node import NodeStatus
from .auth import get_current_user

router = APIRouter()


class DashboardStats(BaseModel):
    """仪表盘统计"""
    customer_count: int
    server_count: int
    server_online_count: int
    socks5_count: int
    socks5_available_count: int
    node_count: int
    node_running_count: int


class ExpiringCustomer(BaseModel):
    """即将到期的客户"""
    id: int
    name: str
    expire_at: datetime
    days_left: int


class AlertItem(BaseModel):
    """告警项"""
    type: str  # customer_expire / node_error / socks5_unavailable
    title: str
    detail: str
    target_id: int


class DashboardResponse(BaseModel):
    """仪表盘响应"""
    stats: DashboardStats
    expiring_customers: List[ExpiringCustomer]
    alerts: List[AlertItem]


@router.get("", response_model=DashboardResponse)
async def get_dashboard(
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取仪表盘数据"""
    
    # 统计数据
    customer_count = await db.scalar(select(func.count(Customer.id)))
    server_count = await db.scalar(select(func.count(Server.id)))
    server_online_count = await db.scalar(
        select(func.count(Server.id)).where(Server.status == ServerStatus.ONLINE)
    )
    socks5_count = await db.scalar(select(func.count(Socks5Proxy.id)))
    socks5_available_count = await db.scalar(
        select(func.count(Socks5Proxy.id)).where(Socks5Proxy.status == Socks5Status.AVAILABLE)
    )
    node_count = await db.scalar(select(func.count(Node.id)))
    node_running_count = await db.scalar(
        select(func.count(Node.id)).where(Node.status == NodeStatus.RUNNING)
    )
    
    stats = DashboardStats(
        customer_count=customer_count or 0,
        server_count=server_count or 0,
        server_online_count=server_online_count or 0,
        socks5_count=socks5_count or 0,
        socks5_available_count=socks5_available_count or 0,
        node_count=node_count or 0,
        node_running_count=node_running_count or 0
    )
    
    # 即将到期的客户（7天内）
    expire_threshold = datetime.now() + timedelta(days=7)
    result = await db.execute(
        select(Customer)
        .where(Customer.expire_at.isnot(None))
        .where(Customer.expire_at <= expire_threshold)
        .where(Customer.expire_at >= datetime.now())
        .order_by(Customer.expire_at)
        .limit(10)
    )
    expiring_customers = []
    for customer in result.scalars():
        days_left = (customer.expire_at - datetime.now()).days
        expiring_customers.append(ExpiringCustomer(
            id=customer.id,
            name=customer.name,
            expire_at=customer.expire_at,
            days_left=days_left
        ))
    
    # 告警
    alerts = []
    
    # 已过期的客户
    result = await db.execute(
        select(Customer)
        .where(Customer.expire_at.isnot(None))
        .where(Customer.expire_at < datetime.now())
        .limit(5)
    )
    for customer in result.scalars():
        alerts.append(AlertItem(
            type="customer_expire",
            title="客户服务已过期",
            detail=f"{customer.name} 服务已于 {customer.expire_at.strftime('%Y-%m-%d')} 到期",
            target_id=customer.id
        ))
    
    # 异常节点
    result = await db.execute(
        select(Node)
        .where(Node.status == NodeStatus.ERROR)
        .limit(5)
    )
    for node in result.scalars():
        alerts.append(AlertItem(
            type="node_error",
            title="节点异常",
            detail=f"节点 {node.name} 状态异常",
            target_id=node.id
        ))
    
    # 不可用的SK5
    result = await db.execute(
        select(Socks5Proxy)
        .where(Socks5Proxy.status == Socks5Status.UNAVAILABLE)
        .limit(5)
    )
    for socks5 in result.scalars():
        alerts.append(AlertItem(
            type="socks5_unavailable",
            title="SK5不可用",
            detail=f"SK5 {socks5.ip}:{socks5.port} 不可用",
            target_id=socks5.id
        ))
    
    return DashboardResponse(
        stats=stats,
        expiring_customers=expiring_customers,
        alerts=alerts
    )




