"""
域名管理API
"""
from typing import Optional
from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy import select, func
from sqlalchemy.ext.asyncio import AsyncSession

from ..core.database import get_db
from ..models import Domain, Customer
from ..schemas.domain import (
    DomainCreate, 
    DomainUpdate, 
    DomainResponse, 
    DomainListResponse
)
from .auth import get_current_user

router = APIRouter()


@router.get("", response_model=DomainListResponse)
async def list_domains(
    page: int = Query(1, ge=1),
    page_size: int = Query(20, ge=1, le=100),
    keyword: Optional[str] = None,
    customer_id: Optional[int] = None,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取域名列表"""
    query = select(Domain)
    count_query = select(func.count(Domain.id))
    
    if keyword:
        query = query.where(Domain.domain.contains(keyword))
        count_query = count_query.where(Domain.domain.contains(keyword))
    
    if customer_id is not None:
        query = query.where(Domain.customer_id == customer_id)
        count_query = count_query.where(Domain.customer_id == customer_id)
    
    total = await db.scalar(count_query)
    
    query = query.order_by(Domain.created_at.desc())
    query = query.offset((page - 1) * page_size).limit(page_size)
    
    result = await db.execute(query)
    domains = result.scalars().all()
    
    items = []
    for domain in domains:
        customer_name = None
        if domain.customer_id:
            customer = await db.get(Customer, domain.customer_id)
            customer_name = customer.name if customer else None
        
        items.append(DomainResponse(
            id=domain.id,
            customer_id=domain.customer_id,
            customer_name=customer_name,
            domain=domain.domain,
            ssl_status=domain.ssl_status,
            ssl_expire_at=domain.ssl_expire_at,
            remark=domain.remark,
            created_at=domain.created_at
        ))
    
    return DomainListResponse(items=items, total=total or 0)


@router.post("", response_model=DomainResponse)
async def create_domain(
    data: DomainCreate,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """创建域名"""
    # 检查域名是否已存在
    existing = await db.scalar(
        select(Domain).where(Domain.domain == data.domain)
    )
    if existing:
        raise HTTPException(status_code=400, detail="域名已存在")
    
    domain = Domain(**data.model_dump())
    db.add(domain)
    await db.commit()
    await db.refresh(domain)
    
    return DomainResponse(
        id=domain.id,
        customer_id=domain.customer_id,
        customer_name=None,
        domain=domain.domain,
        ssl_status=domain.ssl_status,
        ssl_expire_at=domain.ssl_expire_at,
        remark=domain.remark,
        created_at=domain.created_at
    )


@router.get("/{domain_id}", response_model=DomainResponse)
async def get_domain(
    domain_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取域名详情"""
    domain = await db.get(Domain, domain_id)
    if not domain:
        raise HTTPException(status_code=404, detail="域名不存在")
    
    customer_name = None
    if domain.customer_id:
        customer = await db.get(Customer, domain.customer_id)
        customer_name = customer.name if customer else None
    
    return DomainResponse(
        id=domain.id,
        customer_id=domain.customer_id,
        customer_name=customer_name,
        domain=domain.domain,
        ssl_status=domain.ssl_status,
        ssl_expire_at=domain.ssl_expire_at,
        remark=domain.remark,
        created_at=domain.created_at
    )


@router.put("/{domain_id}", response_model=DomainResponse)
async def update_domain(
    domain_id: int,
    data: DomainUpdate,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """更新域名"""
    domain = await db.get(Domain, domain_id)
    if not domain:
        raise HTTPException(status_code=404, detail="域名不存在")
    
    update_data = data.model_dump(exclude_unset=True)
    
    # 检查新域名是否与其他记录冲突
    if "domain" in update_data:
        existing = await db.scalar(
            select(Domain).where(
                Domain.domain == update_data["domain"],
                Domain.id != domain_id
            )
        )
        if existing:
            raise HTTPException(status_code=400, detail="域名已存在")
    
    for key, value in update_data.items():
        setattr(domain, key, value)
    
    await db.commit()
    await db.refresh(domain)
    
    customer_name = None
    if domain.customer_id:
        customer = await db.get(Customer, domain.customer_id)
        customer_name = customer.name if customer else None
    
    return DomainResponse(
        id=domain.id,
        customer_id=domain.customer_id,
        customer_name=customer_name,
        domain=domain.domain,
        ssl_status=domain.ssl_status,
        ssl_expire_at=domain.ssl_expire_at,
        remark=domain.remark,
        created_at=domain.created_at
    )


@router.delete("/{domain_id}")
async def delete_domain(
    domain_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """删除域名"""
    domain = await db.get(Domain, domain_id)
    if not domain:
        raise HTTPException(status_code=404, detail="域名不存在")
    
    await db.delete(domain)
    await db.commit()
    
    return {"message": "删除成功"}




