# -*- coding: utf-8 -*-
"""
🔐 许可证API - 客户端接口（只有激活和验证功能）

⚠️ 安全说明:
- 此模块只提供激活和验证功能
- 卡密生成需要使用管理员本地的 tools/license_generator.py
- 客户无法通过此API生成或管理卡密
"""

from fastapi import APIRouter, HTTPException, Depends
from pydantic import BaseModel
from typing import Optional
from sqlalchemy.ext.asyncio import AsyncSession

from ..core.license_verifier import get_license_verifier, LicenseInfo
from ..core.database import get_db

router = APIRouter()


# ==================== 请求/响应模型 ====================

class ActivateRequest(BaseModel):
    """激活请求"""
    license_key: str


class ActivateResponse(BaseModel):
    """激活响应"""
    success: bool
    message: str
    license_type: Optional[str] = None
    expire_date: Optional[str] = None
    days_remaining: Optional[int] = None


class VerifyResponse(BaseModel):
    """验证响应"""
    is_valid: bool
    is_activated: bool
    license_type: str
    type_name: str
    machine_code: str
    activated_date: Optional[str] = None
    expire_date: Optional[str] = None
    days_remaining: int
    message: str


class MachineCodeResponse(BaseModel):
    """机器码响应"""
    machine_code: str


# ==================== API接口 ====================

@router.post("/activate", response_model=ActivateResponse, summary="激活许可证")
async def activate_license(request: ActivateRequest):
    """
    激活许可证
    
    使用从管理员处获得的卡密激活软件。
    首次激活会绑定当前设备，之后只能在此设备使用。
    """
    verifier = get_license_verifier()
    
    success, message = verifier.activate(request.license_key)
    
    if success:
        info = verifier.verify()
        return ActivateResponse(
            success=True,
            message=message,
            license_type=info.license_type.value,
            expire_date=info.expire_date,
            days_remaining=info.days_remaining
        )
    else:
        return ActivateResponse(
            success=False,
            message=message
        )


@router.get("/verify", response_model=VerifyResponse, summary="验证许可证状态")
async def verify_license():
    """
    验证当前许可证状态
    
    返回许可证的有效性、类型、剩余天数等信息。
    """
    verifier = get_license_verifier()
    info = verifier.verify()
    
    type_names = {
        "trial": "试用版",
        "starter_monthly": "🥉 创业版（月付）",
        "starter_yearly": "🥉 创业版（年付）",
        "matrix_monthly": "🥇 矩阵版（月付）",
        "matrix_yearly": "🥇 矩阵版（年付）",
        "agency_monthly": "💎 机构版（月付）",
        "agency_yearly": "💎 机构版（年付）"
    }
    
    return VerifyResponse(
        is_valid=info.is_valid,
        is_activated=info.is_bound,
        license_type=info.license_type.value,
        type_name=type_names.get(info.license_type.value, "未知"),
        machine_code=info.machine_code,
        activated_date=info.activated_date or None,
        expire_date=info.expire_date,
        days_remaining=info.days_remaining,
        message=info.message
    )


@router.get("/machine-code", response_model=MachineCodeResponse, summary="获取机器码")
async def get_machine_code():
    """
    获取当前设备的机器码
    
    如需联系客服解绑或换绑设备，请提供此机器码。
    """
    verifier = get_license_verifier()
    return MachineCodeResponse(
        machine_code=verifier.get_machine_code()
    )


@router.get("/status", summary="快速检查许可证状态")
async def check_status():
    """
    快速检查许可证状态
    
    返回简单的布尔值，用于前端快速判断是否需要激活。
    """
    verifier = get_license_verifier()
    info = verifier.verify()
    
    # 获取到期提醒
    warning = verifier.get_expiry_warning()
    
    return {
        "activated": info.is_bound,
        "valid": info.is_valid,
        "days_remaining": info.days_remaining if info.is_valid else 0,
        "warning": warning
    }


@router.get("/check-node-limit", summary="检查节点创建限制")
async def check_node_limit(db: AsyncSession = Depends(get_db)):
    """
    检查是否可以创建节点（试用版限制）
    """
    from sqlalchemy import select, func
    from ..models.node import Node
    
    # 查询当前节点总数
    count_query = select(func.count(Node.id))
    current_node_count = await db.scalar(count_query) or 0
    
    verifier = get_license_verifier()
    can_create, message = verifier.can_create_node(current_node_count)
    remaining = verifier.get_daily_remaining()
    
    # 获取试用版限制信息
    info = verifier.verify()
    limits = verifier.VERSION_LIMITS.get(info.license_type, {})
    max_nodes = limits.get("max_nodes", -1)
    
    return {
        "can_create": can_create,
        "message": message,
        "daily_remaining": remaining,
        "current_nodes": current_node_count,
        "max_nodes": max_nodes  # -1 表示无限制
    }


@router.post("/enforce-restrictions", summary="检查许可证状态（已弃用）")
async def enforce_restrictions(db: AsyncSession = Depends(get_db)):
    """
    检查许可证状态（不再自动停止节点）
    
    设计原则：节点一旦部署就应该持续运行，不因许可证问题中断服务。
    参考 3x-ui 的设计，节点稳定性优先。
    """
    verifier = get_license_verifier()
    info = verifier.verify()
    
    return {
        "enforced": False,
        "message": "许可证状态检查完成，节点不会被自动停止",
        "is_valid": info.is_valid,
        "days_remaining": info.days_remaining if info.is_valid else 0,
        "license_type": info.license_type
    }


# ==================== 版本检查和更新 ====================

@router.get("/version-check", summary="检查版本更新")
async def check_version():
    """
    检查是否有新版本
    
    返回：
    - has_update: 是否有更新
    - latest_version: 最新版本号
    - current_version: 当前版本号
    - force_update: 是否强制更新
    - download_url: 下载地址
    - update_message: 更新说明
    """
    from ..core.config import settings
    
    current_version = settings.APP_VERSION
    
    # TODO: 实际应该从远程服务器获取最新版本信息
    # 这里返回一个模拟数据，您可以根据需要修改
    
    # 示例：从您的服务器获取最新版本信息
    # import httpx
    # try:
    #     async with httpx.AsyncClient() as client:
    #         response = await client.get(settings.UPDATE_CHECK_URL, timeout=5)
    #         data = response.json()
    #         latest_version = data.get("version")
    #         update_message = data.get("message")
    # except:
    #     # 网络错误，返回无更新
    #     return {
    #         "has_update": False,
    #         "current_version": current_version
    #     }
    
    # 模拟数据（暂时返回无更新）
    latest_version = current_version
    has_update = False
    force_update = False
    
    # 如果您需要推送更新，修改这里：
    # latest_version = "1.1.0"  # 最新版本
    # has_update = True
    # force_update = True  # 强制更新
    
    return {
        "has_update": has_update,
        "latest_version": latest_version,
        "current_version": current_version,
        "force_update": force_update,
        "download_url": settings.DOWNLOAD_PAGE_URL if has_update else None,
        "update_message": "发现新版本，请及时更新以获得最新功能和安全修复！" if has_update else "当前已是最新版本",
        "changelog": [
            # 您可以在这里添加更新日志
            # "新增：支持XX协议",
            # "修复：XX问题",
            # "优化：XX性能"
        ] if has_update else []
    }
