"""
日志管理 API
"""
from fastapi import APIRouter, Depends, Query, HTTPException
from fastapi.responses import PlainTextResponse, FileResponse
from pydantic import BaseModel
from typing import List, Optional
from datetime import datetime

from .auth import get_current_user
from ..core.logger import get_log_manager

router = APIRouter(prefix="/logs", tags=["日志管理"])


class LogFileInfo(BaseModel):
    """日志文件信息"""
    name: str
    path: str
    size: int
    size_human: str
    modified: str


class LogListResponse(BaseModel):
    """日志文件列表响应"""
    files: List[LogFileInfo]
    log_dir: str


class LogContentResponse(BaseModel):
    """日志内容响应"""
    filename: str
    lines: List[str]
    total_lines: int
    level_filter: Optional[str]


class LogSearchResult(BaseModel):
    """日志搜索结果"""
    line_number: int
    content: str
    level: str
    timestamp: str


@router.get("/files", response_model=LogListResponse)
async def list_log_files(
    current_user: dict = Depends(get_current_user)
):
    """获取所有日志文件列表"""
    manager = get_log_manager()
    files = manager.get_log_files()
    
    return LogListResponse(
        files=files,
        log_dir=str(manager.log_dir)
    )


@router.get("/content/{filename}")
async def get_log_content(
    filename: str,
    lines: int = Query(default=200, ge=1, le=5000, description="返回的行数"),
    level: Optional[str] = Query(default=None, description="日志级别过滤"),
    current_user: dict = Depends(get_current_user)
):
    """读取日志文件内容"""
    # 安全检查 - 防止路径遍历
    if ".." in filename or "/" in filename or "\\" in filename:
        raise HTTPException(status_code=400, detail="非法文件名")
    
    manager = get_log_manager()
    content = manager.read_log_file(filename, lines, level)
    
    return LogContentResponse(
        filename=filename,
        lines=content,
        total_lines=len(content),
        level_filter=level
    )


@router.get("/download/{filename}")
async def download_log_file(
    filename: str,
    current_user: dict = Depends(get_current_user)
):
    """下载日志文件"""
    if ".." in filename or "/" in filename or "\\" in filename:
        raise HTTPException(status_code=400, detail="非法文件名")
    
    manager = get_log_manager()
    log_file = manager.log_dir / filename
    
    if not log_file.exists():
        raise HTTPException(status_code=404, detail="日志文件不存在")
    
    return FileResponse(
        path=str(log_file),
        filename=filename,
        media_type="text/plain"
    )


@router.get("/search")
async def search_logs(
    keyword: str = Query(..., min_length=1, description="搜索关键词"),
    filename: Optional[str] = Query(default=None, description="指定日志文件"),
    level: Optional[str] = Query(default=None, description="日志级别"),
    limit: int = Query(default=100, ge=1, le=500, description="返回结果数量"),
    current_user: dict = Depends(get_current_user)
):
    """搜索日志内容"""
    manager = get_log_manager()
    results = []
    
    # 确定要搜索的文件
    if filename:
        if ".." in filename or "/" in filename or "\\" in filename:
            raise HTTPException(status_code=400, detail="非法文件名")
        files_to_search = [filename]
    else:
        files_to_search = [f['name'] for f in manager.get_log_files() if not f['name'].endswith('.json.log')]
    
    for fname in files_to_search:
        lines = manager.read_log_file(fname, lines=5000, level=level)
        for i, line in enumerate(lines):
            if keyword.lower() in line.lower():
                # 解析日志行
                parts = line.split(" | ")
                timestamp = parts[0] if len(parts) > 0 else ""
                log_level = parts[1].strip() if len(parts) > 1 else "UNKNOWN"
                
                results.append({
                    "file": fname,
                    "line_number": i + 1,
                    "content": line,
                    "level": log_level,
                    "timestamp": timestamp
                })
                
                if len(results) >= limit:
                    break
        
        if len(results) >= limit:
            break
    
    return {
        "keyword": keyword,
        "total_results": len(results),
        "results": results
    }


@router.delete("/clear")
async def clear_old_logs(
    days: int = Query(default=30, ge=1, le=365, description="保留最近多少天的日志"),
    current_user: dict = Depends(get_current_user)
):
    """清理旧日志文件"""
    manager = get_log_manager()
    count = manager.clear_old_logs(days)
    
    return {
        "message": f"已清理 {count} 个旧日志文件",
        "retained_days": days
    }


@router.get("/stats")
async def get_log_stats(
    current_user: dict = Depends(get_current_user)
):
    """获取日志统计信息"""
    manager = get_log_manager()
    files = manager.get_log_files()
    
    total_size = sum(f['size'] for f in files)
    
    # 统计最近的错误数量
    error_count = 0
    warning_count = 0
    
    error_lines = manager.read_log_file("error.log", lines=1000)
    error_count = len(error_lines)
    
    main_lines = manager.read_log_file("app.log", lines=2000)
    for line in main_lines:
        if "| WARNING " in line:
            warning_count += 1
    
    return {
        "total_files": len(files),
        "total_size": total_size,
        "total_size_human": manager._format_size(total_size),
        "log_dir": str(manager.log_dir),
        "recent_errors": error_count,
        "recent_warnings": warning_count,
        "files": files
    }

