"""
节点管理API
"""
from typing import Optional
from datetime import datetime
from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy import select, func
from sqlalchemy.ext.asyncio import AsyncSession

from ..core.database import get_db
from ..models import Node, Customer, Server, Socks5Proxy, Domain
from ..models.node import NodeStatus
from ..schemas.node import (
    NodeDeployRequest, 
    NodeResponse, 
    NodeListResponse,
    NodeShareInfo,
    NodeSettingsRequest,
    NodeTrafficInfo,
    NodeResetTrafficRequest
)
from ..services.deploy_service import DeployService
from ..services.link_service import LinkService
from ..services.traffic_service import TrafficService
from .auth import get_current_user
from ..core.license_verifier import get_license_verifier

router = APIRouter()


def build_node_response(node: Node, customer: Customer, server: Server, 
                        socks5: Socks5Proxy, domain: Optional[Domain]) -> NodeResponse:
    """构建节点响应"""
    return NodeResponse(
        id=node.id,
        customer_id=node.customer_id,
        customer_name=customer.name if customer else "未知",
        server_id=node.server_id,
        server_name=server.name if server else "未知",
        server_ip=server.ip if server else "未知",
        socks5_id=node.socks5_id,
        socks5_info=f"{socks5.ip}:{socks5.port}" if socks5 else "未知",
        domain_id=node.domain_id,
        domain_name=domain.domain if domain else None,
        name=node.name,
        protocol=node.protocol,
        listen_port=node.listen_port,
        uuid=node.uuid,
        reality_dest=node.reality_dest,
        reality_public_key=node.reality_public_key,
        reality_short_id=node.reality_short_id,
        share_link=node.share_link,
        subscribe_token=node.subscribe_token,
        status=node.status,
        latency=node.latency,
        last_check=node.last_check,
        # 流量统计
        traffic_up=node.traffic_up,
        traffic_down=node.traffic_down,
        traffic_total=node.traffic_total,
        traffic_limit=node.traffic_limit,
        last_traffic_sync=node.last_traffic_sync,
        # 到期信息
        expire_at=node.expire_at,
        is_expired=node.is_expired,
        is_traffic_exceeded=node.is_traffic_exceeded,
        created_at=node.created_at
    )


@router.get("", response_model=NodeListResponse)
async def list_nodes(
    page: int = Query(1, ge=1),
    page_size: int = Query(20, ge=1, le=100),
    keyword: Optional[str] = None,
    customer_id: Optional[int] = None,
    server_id: Optional[int] = None,
    status: Optional[str] = None,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取节点列表"""
    from sqlalchemy import or_
    
    query = select(Node)
    count_query = select(func.count(Node.id))
    
    # 关键词搜索：支持节点名称和出口IP
    if keyword:
        # 先查询匹配关键词的SK5代理ID
        socks5_query = select(Socks5Proxy.id).where(Socks5Proxy.ip.contains(keyword))
        socks5_result = await db.execute(socks5_query)
        matching_socks5_ids = [row[0] for row in socks5_result.fetchall()]
        
        # 同时搜索节点名称或出口IP
        if matching_socks5_ids:
            keyword_filter = or_(
                Node.name.contains(keyword),
                Node.socks5_id.in_(matching_socks5_ids)
            )
        else:
            keyword_filter = Node.name.contains(keyword)
        
        query = query.where(keyword_filter)
        count_query = count_query.where(keyword_filter)
    
    if customer_id is not None:
        query = query.where(Node.customer_id == customer_id)
        count_query = count_query.where(Node.customer_id == customer_id)
    
    if server_id is not None:
        query = query.where(Node.server_id == server_id)
        count_query = count_query.where(Node.server_id == server_id)
    
    if status:
        query = query.where(Node.status == status)
        count_query = count_query.where(Node.status == status)
    
    total = await db.scalar(count_query)
    
    query = query.order_by(Node.created_at.desc())
    query = query.offset((page - 1) * page_size).limit(page_size)
    
    result = await db.execute(query)
    nodes = result.scalars().all()
    
    items = []
    for node in nodes:
        customer = await db.get(Customer, node.customer_id)
        server = await db.get(Server, node.server_id)
        socks5 = await db.get(Socks5Proxy, node.socks5_id)
        domain = await db.get(Domain, node.domain_id) if node.domain_id else None
        
        items.append(build_node_response(node, customer, server, socks5, domain))
    
    return NodeListResponse(items=items, total=total or 0)


@router.post("/deploy", response_model=NodeResponse)
async def deploy_node(
    data: NodeDeployRequest,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """部署节点"""
    # 【商业化】检查节点创建限制
    # 先查询当前节点总数（用于试用版限制）
    count_query = select(func.count(Node.id))
    current_node_count = await db.scalar(count_query) or 0
    
    verifier = get_license_verifier()
    can_create, message = verifier.can_create_node(current_node_count)
    if not can_create:
        raise HTTPException(status_code=403, detail=message)
    
    # 验证关联数据
    customer = await db.get(Customer, data.customer_id)
    if not customer:
        raise HTTPException(status_code=404, detail="客户不存在")
    
    server = await db.get(Server, data.server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    socks5 = await db.get(Socks5Proxy, data.socks5_id)
    if not socks5:
        raise HTTPException(status_code=404, detail="SK5不存在")
    
    domain = None
    if data.domain_id:
        domain = await db.get(Domain, data.domain_id)
        if not domain:
            raise HTTPException(status_code=404, detail="域名不存在")
    
    # 部署节点
    deploy_service = DeployService(db)
    try:
        node = await deploy_service.deploy(
            customer=customer,
            server=server,
            socks5=socks5,
            domain=domain,
            protocol=data.protocol,
            listen_port=data.listen_port,
            name=data.name,
            reality_dest=data.reality_dest,
            reality_server_names=data.reality_server_names
        )
        
        # 【商业化】记录节点创建（试用版限制）
        verifier.record_node_creation()
        
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"部署失败: {str(e)}")
    
    return build_node_response(node, customer, server, socks5, domain)


# ========== 批量操作路由（必须在 /{node_id} 之前定义）==========

@router.post("/traffic/sync-all")
async def sync_all_traffic(
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """批量同步所有节点流量"""
    traffic_service = TrafficService(db)
    result = await traffic_service.sync_all_nodes_traffic()
    
    return {
        "message": f"同步完成: 成功 {result['success']} 个, 失败 {result['failed']} 个, 跳过 {result['skipped']} 个",
        **result
    }


@router.post("/check-limits")
async def check_limits(
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """检查并停止超限/过期节点"""
    traffic_service = TrafficService(db)
    stopped_nodes = await traffic_service.check_and_stop_exceeded_nodes()
    
    return {
        "message": f"检查完成，已停止 {len(stopped_nodes)} 个节点",
        "stopped_nodes": stopped_nodes
    }


@router.post("/restart-all")
async def restart_all_nodes(
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """一键恢复所有已停止的节点（续费后使用）"""
    from sqlalchemy import select
    
    # 查找所有已停止的节点
    query = select(Node).where(Node.status == NodeStatus.STOPPED)
    result = await db.execute(query)
    nodes = result.scalars().all()
    
    if not nodes:
        return {
            "message": "没有需要恢复的节点",
            "success": 0,
            "failed": 0
        }
    
    deploy_service = DeployService(db)
    success_count = 0
    failed_count = 0
    failed_nodes = []
    
    for node in nodes:
        try:
            server = await db.get(Server, node.server_id)
            if server:
                await deploy_service.restart_node(server, node)
                node.status = NodeStatus.RUNNING
                success_count += 1
        except Exception as e:
            failed_count += 1
            failed_nodes.append({"id": node.id, "name": node.name, "error": str(e)})
    
    await db.commit()
    
    return {
        "message": f"恢复完成: 成功 {success_count} 个, 失败 {failed_count} 个",
        "success": success_count,
        "failed": failed_count,
        "failed_nodes": failed_nodes
    }


@router.post("/restore-all-config")
async def restore_all_nodes_config(
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """
    恢复所有节点配置 - 从备份恢复旧配置
    """
    from sqlalchemy import select
    
    # 查找所有节点
    query = select(Node)
    result = await db.execute(query)
    nodes = result.scalars().all()
    
    if not nodes:
        return {
            "message": "没有节点需要恢复",
            "success": 0,
            "failed": 0
        }
    
    deploy_service = DeployService(db)
    success_count = 0
    failed_count = 0
    failed_nodes = []
    
    for node in nodes:
        try:
            server = await db.get(Server, node.server_id)
            
            if server:
                await deploy_service.restore_node_config(node, server)
                success_count += 1
            else:
                failed_count += 1
                failed_nodes.append({
                    "id": node.id, 
                    "name": node.name, 
                    "error": "缺少服务器配置"
                })
        except Exception as e:
            failed_count += 1
            failed_nodes.append({"id": node.id, "name": node.name, "error": str(e)})
    
    return {
        "message": f"恢复完成: 成功 {success_count} 个, 失败 {failed_count} 个",
        "success": success_count,
        "failed": failed_count,
        "failed_nodes": failed_nodes
    }


# ========== 单个节点操作路由 ==========

@router.get("/{node_id}", response_model=NodeResponse)
async def get_node(
    node_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取节点详情"""
    node = await db.get(Node, node_id)
    if not node:
        raise HTTPException(status_code=404, detail="节点不存在")
    
    customer = await db.get(Customer, node.customer_id)
    server = await db.get(Server, node.server_id)
    socks5 = await db.get(Socks5Proxy, node.socks5_id)
    domain = await db.get(Domain, node.domain_id) if node.domain_id else None
    
    return build_node_response(node, customer, server, socks5, domain)


@router.get("/{node_id}/share", response_model=NodeShareInfo)
async def get_node_share(
    node_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取节点分享信息"""
    node = await db.get(Node, node_id)
    if not node:
        raise HTTPException(status_code=404, detail="节点不存在")
    
    link_service = LinkService()
    qrcode_base64 = link_service.generate_qrcode(node.share_link)
    
    return NodeShareInfo(
        share_link=node.share_link or "",
        qrcode_base64=qrcode_base64,
        config_json=node.config_json or ""
    )


@router.post("/{node_id}/restart")
async def restart_node(
    node_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """重启节点"""
    node = await db.get(Node, node_id)
    if not node:
        raise HTTPException(status_code=404, detail="节点不存在")
    
    server = await db.get(Server, node.server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    deploy_service = DeployService(db)
    try:
        await deploy_service.restart_node(server, node)
        node.status = NodeStatus.RUNNING
        await db.commit()
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"重启失败: {str(e)}")
    
    return {"message": "重启成功"}


@router.post("/{node_id}/stop")
async def stop_node(
    node_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """停止节点"""
    node = await db.get(Node, node_id)
    if not node:
        raise HTTPException(status_code=404, detail="节点不存在")
    
    server = await db.get(Server, node.server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    deploy_service = DeployService(db)
    try:
        await deploy_service.stop_node(server, node)
        node.status = NodeStatus.STOPPED
        await db.commit()
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"停止失败: {str(e)}")
    
    return {"message": "停止成功"}


@router.delete("/{node_id}")
async def delete_node(
    node_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """删除节点"""
    node = await db.get(Node, node_id)
    if not node:
        raise HTTPException(status_code=404, detail="节点不存在")
    
    server = await db.get(Server, node.server_id)
    
    # 尝试停止远程服务
    if server:
        deploy_service = DeployService(db)
        try:
            await deploy_service.stop_node(server, node)
        except Exception:
            pass  # 忽略停止失败
    
    await db.delete(node)
    await db.commit()
    
    return {"message": "删除成功"}


@router.get("/{node_id}/traffic", response_model=NodeTrafficInfo)
async def get_node_traffic(
    node_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取节点流量信息"""
    node = await db.get(Node, node_id)
    if not node:
        raise HTTPException(status_code=404, detail="节点不存在")
    
    traffic_service = TrafficService(db)
    
    # 计算流量使用百分比
    traffic_percent = traffic_service.calculate_traffic_percent(
        node.traffic_total, 
        node.traffic_limit
    )
    
    # 计算剩余天数
    days_remaining = traffic_service.calculate_days_remaining(node.expire_at)
    
    return NodeTrafficInfo(
        traffic_up=node.traffic_up,
        traffic_down=node.traffic_down,
        traffic_total=node.traffic_total,
        traffic_limit=node.traffic_limit,
        traffic_used_percent=traffic_percent,
        expire_at=node.expire_at,
        days_remaining=days_remaining,
        is_expired=node.is_expired,
        is_traffic_exceeded=node.is_traffic_exceeded,
        last_sync=node.last_traffic_sync
    )


@router.post("/{node_id}/traffic/sync")
async def sync_node_traffic(
    node_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """同步节点流量（从服务器获取最新数据）"""
    node = await db.get(Node, node_id)
    if not node:
        raise HTTPException(status_code=404, detail="节点不存在")
    
    server = await db.get(Server, node.server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    traffic_service = TrafficService(db)
    success = await traffic_service.sync_node_traffic(node, server)
    
    if not success:
        return {
            "message": "同步失败，可能需要重新部署节点以启用流量统计",
            "success": False
        }
    
    return {
        "message": "同步成功",
        "success": True,
        "traffic_up": node.traffic_up,
        "traffic_down": node.traffic_down,
        "traffic_total": node.traffic_total
    }


@router.post("/{node_id}/traffic/reset")
async def reset_node_traffic(
    node_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """重置节点流量统计"""
    node = await db.get(Node, node_id)
    if not node:
        raise HTTPException(status_code=404, detail="节点不存在")
    
    server = await db.get(Server, node.server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    traffic_service = TrafficService(db)
    await traffic_service.reset_node_traffic(node, server)
    
    return {"message": "流量已重置"}


@router.get("/{node_id}/traffic/debug")
async def debug_node_traffic(
    node_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """调试节点流量统计 - 返回详细信息"""
    from ..services.ssh_service import SSHService
    from ..core.security import decrypt_string
    
    node = await db.get(Node, node_id)
    if not node:
        raise HTTPException(status_code=404, detail="节点不存在")
    
    server = await db.get(Server, node.server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    server_password = decrypt_string(server.ssh_password)
    if not server_password:
        server_password = server.ssh_password
    
    ssh_service = SSHService()
    service_name = f"xray-node-{node_id}"
    
    # 调试脚本
    debug_script = f'''#!/bin/bash
echo "=== 流量统计调试信息 ==="
echo ""

# 1. 检查Xray路径
echo "1. Xray路径检查:"
XRAY_BIN=""
for path in /usr/local/bin/xray /usr/bin/xray /opt/xray/xray; do
    if [ -x "$path" ]; then
        XRAY_BIN="$path"
        echo "   找到Xray: $XRAY_BIN"
        echo "   版本: $($XRAY_BIN version 2>/dev/null | head -1)"
        break
    fi
done
[ -z "$XRAY_BIN" ] && echo "   未找到Xray"
echo ""

# 2. 检查服务状态
echo "2. 服务状态检查:"
echo "   服务名: {service_name}"
systemctl is-active --quiet {service_name} && echo "   状态: 运行中" || echo "   状态: 未运行"
echo ""

# 3. 检查API端口
echo "3. API端口检查 (10085):"
if ss -tlnp 2>/dev/null | grep -q ":10085"; then
    echo "   端口10085: 正在监听"
    ss -tlnp 2>/dev/null | grep ":10085"
else
    echo "   端口10085: 未监听"
fi
echo ""

# 4. 检查配置文件
echo "4. 配置文件检查:"
CONFIG_FILE="/etc/xray/config_{node_id}.json"
if [ -f "$CONFIG_FILE" ]; then
    echo "   配置文件存在: $CONFIG_FILE"
    echo "   stats配置: $(grep -c '"stats"' $CONFIG_FILE) 处"
    echo "   api配置: $(grep -c '"api"' $CONFIG_FILE) 处"
else
    echo "   配置文件不存在"
fi
echo ""

# 5. 尝试获取流量
echo "5. 流量获取测试:"
if [ -n "$XRAY_BIN" ]; then
    echo "   上行流量:"
    $XRAY_BIN api stats --server=127.0.0.1:10085 -name "inbound>>>vless-in>>>traffic>>>uplink" 2>&1 || echo "   获取失败"
    echo ""
    echo "   下行流量:"
    $XRAY_BIN api stats --server=127.0.0.1:10085 -name "inbound>>>vless-in>>>traffic>>>downlink" 2>&1 || echo "   获取失败"
fi
echo ""

# 6. 列出所有可用统计项
echo "6. 所有可用统计项:"
if [ -n "$XRAY_BIN" ]; then
    $XRAY_BIN api statsquery --server=127.0.0.1:10085 2>&1 | head -50 || echo "   查询失败"
fi

echo ""
echo "=== 调试完成 ==="
'''
    
    success, stdout, stderr = await ssh_service.execute_script(
        host=server.ip,
        port=server.ssh_port,
        username=server.ssh_user,
        password=server_password,
        script_content=debug_script,
        timeout=30
    )
    
    return {
        "node_id": node_id,
        "node_name": node.name,
        "server_ip": server.ip,
        "success": success,
        "output": stdout,
        "error": stderr
    }


@router.put("/{node_id}/settings")
async def update_node_settings(
    node_id: int,
    data: NodeSettingsRequest,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """更新节点设置（流量限制、到期时间）"""
    node = await db.get(Node, node_id)
    if not node:
        raise HTTPException(status_code=404, detail="节点不存在")
    
    # 更新流量限制
    if data.traffic_limit is not None:
        node.traffic_limit = data.traffic_limit if data.traffic_limit > 0 else None
        # 检查是否超限
        if node.traffic_limit and node.traffic_total >= node.traffic_limit:
            node.is_traffic_exceeded = True
        else:
            node.is_traffic_exceeded = False
    
    # 更新到期时间
    if data.expire_at is not None:
        node.expire_at = data.expire_at
        # 检查是否过期
        if node.expire_at and datetime.now() >= node.expire_at:
            node.is_expired = True
        else:
            node.is_expired = False
    
    await db.commit()
    
    return {"message": "设置已更新"}
