"""
服务器管理API
"""
from typing import Optional
from datetime import datetime
from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy import select, func
from sqlalchemy.ext.asyncio import AsyncSession

from ..core.database import get_db
from ..core.security import encrypt_string, decrypt_string
from ..models import Server, Customer, Node
from ..models.server import ServerStatus
from ..schemas.server import (
    ServerCreate, 
    ServerUpdate, 
    ServerResponse, 
    ServerListResponse,
    ServerParseRequest,
    ServerParseResponse
)
from ..schemas.common import TestResult
from ..services.ssh_service import SSHService
from ..services.parser_service import ParserService
from .auth import get_current_user

router = APIRouter()


@router.get("", response_model=ServerListResponse)
async def list_servers(
    page: int = Query(1, ge=1),
    page_size: int = Query(20, ge=1, le=100),
    keyword: Optional[str] = None,
    customer_id: Optional[int] = None,
    is_own: Optional[bool] = None,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取服务器列表"""
    query = select(Server)
    count_query = select(func.count(Server.id))
    
    if keyword:
        query = query.where(
            (Server.name.contains(keyword)) | (Server.ip.contains(keyword))
        )
        count_query = count_query.where(
            (Server.name.contains(keyword)) | (Server.ip.contains(keyword))
        )
    
    if customer_id is not None:
        query = query.where(Server.customer_id == customer_id)
        count_query = count_query.where(Server.customer_id == customer_id)
    
    if is_own is not None:
        query = query.where(Server.is_own == is_own)
        count_query = count_query.where(Server.is_own == is_own)
    
    total = await db.scalar(count_query)
    
    query = query.order_by(Server.created_at.desc())
    query = query.offset((page - 1) * page_size).limit(page_size)
    
    result = await db.execute(query)
    servers = result.scalars().all()
    
    items = []
    for server in servers:
        customer_name = None
        if server.customer_id:
            customer = await db.get(Customer, server.customer_id)
            customer_name = customer.name if customer else None
        
        node_count = await db.scalar(
            select(func.count(Node.id)).where(Node.server_id == server.id)
        )
        
        items.append(ServerResponse(
            id=server.id,
            customer_id=server.customer_id,
            customer_name=customer_name,
            name=server.name,
            ip=server.ip,
            ssh_port=server.ssh_port,
            ssh_user=server.ssh_user,
            region=server.region,
            status=server.status,
            is_own=server.is_own,
            last_check=server.last_check,
            created_at=server.created_at,
            node_count=node_count or 0
        ))
    
    return ServerListResponse(items=items, total=total or 0)


@router.post("", response_model=ServerResponse)
async def create_server(
    data: ServerCreate,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """创建服务器"""
    # 加密密码
    encrypted_password = encrypt_string(data.ssh_password)
    
    server = Server(
        customer_id=data.customer_id,
        name=data.name,
        ip=data.ip,
        ssh_port=data.ssh_port,
        ssh_user=data.ssh_user,
        ssh_password=encrypted_password,
        region=data.region,
        is_own=data.is_own
    )
    db.add(server)
    await db.commit()
    await db.refresh(server)
    
    return ServerResponse(
        id=server.id,
        customer_id=server.customer_id,
        customer_name=None,
        name=server.name,
        ip=server.ip,
        ssh_port=server.ssh_port,
        ssh_user=server.ssh_user,
        region=server.region,
        status=server.status,
        is_own=server.is_own,
        last_check=server.last_check,
        created_at=server.created_at,
        node_count=0
    )


@router.get("/{server_id}", response_model=ServerResponse)
async def get_server(
    server_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取服务器详情"""
    server = await db.get(Server, server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    customer_name = None
    if server.customer_id:
        customer = await db.get(Customer, server.customer_id)
        customer_name = customer.name if customer else None
    
    node_count = await db.scalar(
        select(func.count(Node.id)).where(Node.server_id == server.id)
    )
    
    return ServerResponse(
        id=server.id,
        customer_id=server.customer_id,
        customer_name=customer_name,
        name=server.name,
        ip=server.ip,
        ssh_port=server.ssh_port,
        ssh_user=server.ssh_user,
        region=server.region,
        status=server.status,
        is_own=server.is_own,
        last_check=server.last_check,
        created_at=server.created_at,
        node_count=node_count or 0
    )


@router.put("/{server_id}", response_model=ServerResponse)
async def update_server(
    server_id: int,
    data: ServerUpdate,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """更新服务器"""
    server = await db.get(Server, server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    update_data = data.model_dump(exclude_unset=True)
    
    # 如果更新密码，需要加密
    if "ssh_password" in update_data and update_data["ssh_password"]:
        update_data["ssh_password"] = encrypt_string(update_data["ssh_password"])
    
    for key, value in update_data.items():
        setattr(server, key, value)
    
    await db.commit()
    await db.refresh(server)
    
    customer_name = None
    if server.customer_id:
        customer = await db.get(Customer, server.customer_id)
        customer_name = customer.name if customer else None
    
    node_count = await db.scalar(
        select(func.count(Node.id)).where(Node.server_id == server.id)
    )
    
    return ServerResponse(
        id=server.id,
        customer_id=server.customer_id,
        customer_name=customer_name,
        name=server.name,
        ip=server.ip,
        ssh_port=server.ssh_port,
        ssh_user=server.ssh_user,
        region=server.region,
        status=server.status,
        is_own=server.is_own,
        last_check=server.last_check,
        created_at=server.created_at,
        node_count=node_count or 0
    )


@router.delete("/{server_id}")
async def delete_server(
    server_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """删除服务器"""
    server = await db.get(Server, server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    # 检查是否有关联节点
    node_count = await db.scalar(
        select(func.count(Node.id)).where(Node.server_id == server.id)
    )
    if node_count and node_count > 0:
        raise HTTPException(status_code=400, detail="该服务器下有节点，无法删除")
    
    await db.delete(server)
    await db.commit()
    
    return {"message": "删除成功"}


@router.post("/{server_id}/test", response_model=TestResult)
async def test_server(
    server_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """测试服务器SSH连接"""
    server = await db.get(Server, server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    password = decrypt_string(server.ssh_password)
    
    ssh_service = SSHService()
    result = await ssh_service.test_connection(
        host=server.ip,
        port=server.ssh_port,
        username=server.ssh_user,
        password=password
    )
    
    # 更新状态
    server.status = ServerStatus.ONLINE if result.success else ServerStatus.OFFLINE
    server.last_check = datetime.now()
    await db.commit()
    
    return result


@router.post("/parse", response_model=ServerParseResponse)
async def parse_server_info(
    data: ServerParseRequest,
    current_user: dict = Depends(get_current_user)
):
    """解析服务器信息文本"""
    result = ParserService.parse_server_info(data.text)
    if not result:
        raise HTTPException(status_code=400, detail="无法解析服务器信息")
    
    return ServerParseResponse(**result)


@router.get("/{server_id}/ports")
async def get_server_ports(
    server_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取服务器已占用的端口"""
    from ..services.deploy_service import DeployService
    
    server = await db.get(Server, server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    deploy_service = DeployService(db)
    try:
        used_ports = await deploy_service.get_used_ports(server)
        
        # 推荐一个可用端口
        recommended_port = await deploy_service.find_available_port(server, 443)
        
        return {
            "used_ports": used_ports,
            "recommended_port": recommended_port,
            "total_used": len(used_ports)
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"获取端口信息失败: {str(e)}")


@router.get("/{server_id}/info")
async def get_server_info(
    server_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取服务器详细信息"""
    from ..services.server_init_service import ServerInitService
    
    server = await db.get(Server, server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    password = decrypt_string(server.ssh_password)
    
    init_service = ServerInitService()
    try:
        info = await init_service.get_server_info(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=password
        )
        return info
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"获取信息失败: {str(e)}")


@router.post("/{server_id}/init")
async def init_server(
    server_id: int,
    steps: list = None,
    new_ssh_port: int = None,
    timezone: str = "Asia/Shanghai",
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """初始化服务器"""
    from ..services.server_init_service import ServerInitService, InitStep
    
    server = await db.get(Server, server_id)
    if not server:
        raise HTTPException(status_code=404, detail="服务器不存在")
    
    password = decrypt_string(server.ssh_password)
    
    # 转换步骤
    init_steps = None
    if steps:
        init_steps = [InitStep(s) for s in steps]
    
    init_service = ServerInitService()
    try:
        results = await init_service.init_server(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=password,
            steps=init_steps,
            new_ssh_port=new_ssh_port,
            timezone=timezone
        )
        
        # 如果修改了SSH端口，更新数据库
        if new_ssh_port:
            server.ssh_port = new_ssh_port
            await db.commit()
        
        return {
            "results": [
                {
                    "step": r.step,
                    "success": r.success,
                    "message": r.message
                }
                for r in results
            ],
            "success_count": sum(1 for r in results if r.success),
            "fail_count": sum(1 for r in results if not r.success)
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"初始化失败: {str(e)}")

