"""
SOCKS5管理API
"""
from typing import Optional, List
from datetime import datetime
from fastapi import APIRouter, Depends, HTTPException, Query
from sqlalchemy import select, func
from sqlalchemy.ext.asyncio import AsyncSession

from ..core.database import get_db
from ..core.security import encrypt_string, decrypt_string
from ..models import Socks5Proxy, Customer, Node
from ..models.socks5 import Socks5Status
from ..schemas.socks5 import (
    Socks5Create, 
    Socks5Update, 
    Socks5Response, 
    Socks5ListResponse,
    Socks5ParseRequest,
    Socks5ParseResponse,
    Socks5BatchCreate
)
from ..schemas.common import TestResult, IPCheckResult
from ..services.socks5_service import Socks5Service
from ..services.parser_service import ParserService
from .auth import get_current_user

router = APIRouter()


@router.get("", response_model=Socks5ListResponse)
async def list_socks5(
    page: int = Query(1, ge=1),
    page_size: int = Query(20, ge=1, le=100),
    keyword: Optional[str] = None,
    customer_id: Optional[int] = None,
    status: Optional[str] = None,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取SK5列表"""
    query = select(Socks5Proxy)
    count_query = select(func.count(Socks5Proxy.id))
    
    if keyword:
        query = query.where(Socks5Proxy.ip.contains(keyword))
        count_query = count_query.where(Socks5Proxy.ip.contains(keyword))
    
    if customer_id is not None:
        query = query.where(Socks5Proxy.customer_id == customer_id)
        count_query = count_query.where(Socks5Proxy.customer_id == customer_id)
    
    if status:
        query = query.where(Socks5Proxy.status == status)
        count_query = count_query.where(Socks5Proxy.status == status)
    
    total = await db.scalar(count_query)
    
    query = query.order_by(Socks5Proxy.created_at.desc())
    query = query.offset((page - 1) * page_size).limit(page_size)
    
    result = await db.execute(query)
    socks5_list = result.scalars().all()
    
    items = []
    for socks5 in socks5_list:
        customer_name = None
        if socks5.customer_id:
            customer = await db.get(Customer, socks5.customer_id)
            customer_name = customer.name if customer else None
        
        node_count = await db.scalar(
            select(func.count(Node.id)).where(Node.socks5_id == socks5.id)
        )
        
        items.append(Socks5Response(
            id=socks5.id,
            customer_id=socks5.customer_id,
            customer_name=customer_name,
            ip=socks5.ip,
            port=socks5.port,
            username=socks5.username,
            region=socks5.region,
            status=socks5.status,
            latency=socks5.latency,
            last_check=socks5.last_check,
            remark=socks5.remark,
            created_at=socks5.created_at,
            node_count=node_count or 0
        ))
    
    return Socks5ListResponse(items=items, total=total or 0)


@router.post("", response_model=Socks5Response)
async def create_socks5(
    data: Socks5Create,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """创建SK5"""
    # 加密密码
    encrypted_password = encrypt_string(data.password) if data.password else None
    
    socks5 = Socks5Proxy(
        customer_id=data.customer_id,
        ip=data.ip,
        port=data.port,
        username=data.username,
        password=encrypted_password,
        remark=data.remark
    )
    db.add(socks5)
    await db.commit()
    await db.refresh(socks5)
    
    return Socks5Response(
        id=socks5.id,
        customer_id=socks5.customer_id,
        customer_name=None,
        ip=socks5.ip,
        port=socks5.port,
        username=socks5.username,
        region=socks5.region,
        status=socks5.status,
        latency=socks5.latency,
        last_check=socks5.last_check,
        remark=socks5.remark,
        created_at=socks5.created_at,
        node_count=0
    )


@router.post("/batch", response_model=List[Socks5Response])
async def batch_create_socks5(
    data: Socks5BatchCreate,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """批量创建SK5"""
    created = []
    
    for item in data.items:
        encrypted_password = encrypt_string(item.password) if item.password else None
        
        socks5 = Socks5Proxy(
            customer_id=data.customer_id,
            ip=item.ip,
            port=item.port,
            username=item.username,
            password=encrypted_password,
            remark=item.remark
        )
        db.add(socks5)
        await db.flush()
        
        created.append(Socks5Response(
            id=socks5.id,
            customer_id=socks5.customer_id,
            customer_name=None,
            ip=socks5.ip,
            port=socks5.port,
            username=socks5.username,
            region=socks5.region,
            status=socks5.status,
            latency=socks5.latency,
            last_check=socks5.last_check,
            remark=socks5.remark,
            created_at=socks5.created_at,
            node_count=0
        ))
    
    await db.commit()
    return created


@router.get("/{socks5_id}", response_model=Socks5Response)
async def get_socks5(
    socks5_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """获取SK5详情"""
    socks5 = await db.get(Socks5Proxy, socks5_id)
    if not socks5:
        raise HTTPException(status_code=404, detail="SK5不存在")
    
    customer_name = None
    if socks5.customer_id:
        customer = await db.get(Customer, socks5.customer_id)
        customer_name = customer.name if customer else None
    
    node_count = await db.scalar(
        select(func.count(Node.id)).where(Node.socks5_id == socks5.id)
    )
    
    return Socks5Response(
        id=socks5.id,
        customer_id=socks5.customer_id,
        customer_name=customer_name,
        ip=socks5.ip,
        port=socks5.port,
        username=socks5.username,
        region=socks5.region,
        status=socks5.status,
        latency=socks5.latency,
        last_check=socks5.last_check,
        remark=socks5.remark,
        created_at=socks5.created_at,
        node_count=node_count or 0
    )


@router.put("/{socks5_id}", response_model=Socks5Response)
async def update_socks5(
    socks5_id: int,
    data: Socks5Update,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """更新SK5"""
    socks5 = await db.get(Socks5Proxy, socks5_id)
    if not socks5:
        raise HTTPException(status_code=404, detail="SK5不存在")
    
    update_data = data.model_dump(exclude_unset=True)
    
    if "password" in update_data and update_data["password"]:
        update_data["password"] = encrypt_string(update_data["password"])
    
    for key, value in update_data.items():
        setattr(socks5, key, value)
    
    await db.commit()
    await db.refresh(socks5)
    
    customer_name = None
    if socks5.customer_id:
        customer = await db.get(Customer, socks5.customer_id)
        customer_name = customer.name if customer else None
    
    node_count = await db.scalar(
        select(func.count(Node.id)).where(Node.socks5_id == socks5.id)
    )
    
    return Socks5Response(
        id=socks5.id,
        customer_id=socks5.customer_id,
        customer_name=customer_name,
        ip=socks5.ip,
        port=socks5.port,
        username=socks5.username,
        region=socks5.region,
        status=socks5.status,
        latency=socks5.latency,
        last_check=socks5.last_check,
        remark=socks5.remark,
        created_at=socks5.created_at,
        node_count=node_count or 0
    )


@router.delete("/{socks5_id}")
async def delete_socks5(
    socks5_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """删除SK5"""
    socks5 = await db.get(Socks5Proxy, socks5_id)
    if not socks5:
        raise HTTPException(status_code=404, detail="SK5不存在")
    
    node_count = await db.scalar(
        select(func.count(Node.id)).where(Node.socks5_id == socks5.id)
    )
    if node_count and node_count > 0:
        raise HTTPException(status_code=400, detail="该SK5下有节点，无法删除")
    
    await db.delete(socks5)
    await db.commit()
    
    return {"message": "删除成功"}


@router.post("/{socks5_id}/test", response_model=TestResult)
async def test_socks5(
    socks5_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """测试SK5连接"""
    socks5 = await db.get(Socks5Proxy, socks5_id)
    if not socks5:
        raise HTTPException(status_code=404, detail="SK5不存在")
    
    password = decrypt_string(socks5.password) if socks5.password else None
    
    socks5_service = Socks5Service()
    result = await socks5_service.test_connection(
        host=socks5.ip,
        port=socks5.port,
        username=socks5.username,
        password=password
    )
    
    # 更新状态
    socks5.status = Socks5Status.AVAILABLE if result.success else Socks5Status.UNAVAILABLE
    socks5.latency = result.latency
    socks5.region = result.region
    socks5.last_check = datetime.now()
    await db.commit()
    
    return result


@router.post("/parse", response_model=Socks5ParseResponse)
async def parse_socks5_info(
    data: Socks5ParseRequest,
    current_user: dict = Depends(get_current_user)
):
    """解析SK5信息文本"""
    result = ParserService.parse_socks5_info(data.text)
    return result


@router.post("/{socks5_id}/check-ip", response_model=IPCheckResult)
async def check_ip_quality(
    socks5_id: int,
    db: AsyncSession = Depends(get_db),
    current_user: dict = Depends(get_current_user)
):
    """
    检测SK5的IP质量和TikTok适用性
    返回IP归属地、类型、纯净度评分等信息
    """
    socks5 = await db.get(Socks5Proxy, socks5_id)
    if not socks5:
        raise HTTPException(status_code=404, detail="SK5不存在")
    
    password = decrypt_string(socks5.password) if socks5.password else None
    
    socks5_service = Socks5Service()
    result = await socks5_service.check_ip_quality(
        host=socks5.ip,
        port=socks5.port,
        username=socks5.username,
        password=password
    )
    
    # 更新SK5的地区信息
    if result.country and result.city:
        socks5.region = f"{result.country} {result.city}"
        await db.commit()
    
    return IPCheckResult(
        ip=result.ip,
        country=result.country,
        country_code=result.country_code,
        region=result.region,
        city=result.city,
        isp=result.isp,
        org=result.org,
        as_name=result.as_name,
        ip_type=result.ip_type,
        is_proxy=result.is_proxy,
        is_vpn=result.is_vpn,
        is_datacenter=result.is_datacenter,
        is_residential=result.is_residential,
        tiktok_score=result.tiktok_score,
        tiktok_risk=result.tiktok_risk,
        risk_reasons=result.risk_reasons
    )

