"""
订阅API - 支持节点订阅和客户级订阅
"""
import base64
from fastapi import APIRouter, Depends, HTTPException, Response
from sqlalchemy import select
from sqlalchemy.ext.asyncio import AsyncSession

from ..core.database import get_db
from ..models import Node, Customer
from ..models.node import NodeStatus

router = APIRouter()


@router.get("/c/{token}")
async def get_customer_subscribe_by_token(
    token: str,
    db: AsyncSession = Depends(get_db)
):
    """
    🌟 客户级订阅 - 一键订阅该客户名下所有节点
    
    使用客户专属的订阅令牌，无需客户ID，更简洁安全
    """
    # 通过token查找客户
    result = await db.execute(
        select(Customer).where(Customer.subscribe_token == token)
    )
    customer = result.scalar_one_or_none()
    
    if not customer:
        raise HTTPException(status_code=404, detail="订阅不存在或已失效")
    
    # 获取该客户的所有运行中节点
    result = await db.execute(
        select(Node).where(
            Node.customer_id == customer.id,
            Node.status == NodeStatus.RUNNING
        ).order_by(Node.created_at.desc())
    )
    nodes = result.scalars().all()
    
    if not nodes:
        raise HTTPException(status_code=404, detail="暂无可用节点")
    
    # 合并所有节点链接
    links = [n.share_link for n in nodes if n.share_link]
    
    if not links:
        raise HTTPException(status_code=404, detail="节点配置未生成")
    
    content = base64.b64encode("\n".join(links).encode()).decode()
    
    # 计算流量信息
    total_up = sum(n.traffic_up or 0 for n in nodes)
    total_down = sum(n.traffic_down or 0 for n in nodes)
    total_limit = sum(n.traffic_limit or 0 for n in nodes if n.traffic_limit)
    
    # 使用URL编码处理中文文件名
    import urllib.parse
    safe_name = urllib.parse.quote(customer.name, safe='')
    
    return Response(
        content=content,
        media_type="text/plain",
        headers={
            "Content-Disposition": f"attachment; filename*=UTF-8''{safe_name}_all.txt",
            "Subscription-Userinfo": f"upload={total_up}; download={total_down}; total={total_limit}; expire=0",
            "Profile-Title": f"base64,{base64.b64encode((customer.name + ' - ' + str(len(links)) + ' nodes').encode()).decode()}"
        }
    )


@router.get("/{token}")
async def get_subscribe(
    token: str,
    db: AsyncSession = Depends(get_db)
):
    """获取单节点订阅内容（无需认证）"""
    # 查找节点
    result = await db.execute(
        select(Node).where(Node.subscribe_token == token)
    )
    node = result.scalar_one_or_none()
    
    if not node:
        raise HTTPException(status_code=404, detail="订阅不存在")
    
    # 返回订阅内容
    if node.share_link:
        import urllib.parse
        content = base64.b64encode(node.share_link.encode()).decode()
        safe_name = urllib.parse.quote(node.name, safe='')
        return Response(
            content=content,
            media_type="text/plain",
            headers={
                "Content-Disposition": f"attachment; filename*=UTF-8''{safe_name}.txt",
                "Subscription-Userinfo": f"upload={node.traffic_up or 0}; download={node.traffic_down or 0}; total={node.traffic_limit or 0}; expire=0"
            }
        )
    
    raise HTTPException(status_code=404, detail="节点配置不存在")




