"""
应用配置
"""
import os
import sys
import secrets
from pathlib import Path
from typing import Optional
from pydantic_settings import BaseSettings
from pydantic import field_validator


def get_base_dir() -> Path:
    """获取程序基础目录（兼容打包环境）"""
    if getattr(sys, 'frozen', False):
        # PyInstaller打包后的环境
        return Path(sys.executable).parent
    else:
        # 开发环境
        return Path(__file__).resolve().parent.parent.parent


# 预先计算路径
_BASE_DIR = get_base_dir()
_DATA_DIR = _BASE_DIR / "data"
_SCRIPTS_DIR = _BASE_DIR / "scripts"

# 确保数据目录存在
_DATA_DIR.mkdir(parents=True, exist_ok=True)

# 数据库URL
_DB_PATH = _DATA_DIR / "app.db"
_DATABASE_URL = f"sqlite+aiosqlite:///{_DB_PATH.as_posix()}"

# 加密密钥
_key_file = _DATA_DIR / ".encryption_key"
if _key_file.exists():
    _ENCRYPTION_KEY = _key_file.read_text().strip()
else:
    _ENCRYPTION_KEY = secrets.token_urlsafe(32)
    _key_file.write_text(_ENCRYPTION_KEY)


class Settings(BaseSettings):
    """应用设置"""
    
    # 应用信息
    APP_NAME: str = "全球节点自动化搭建工具"
    APP_VERSION: str = "2.3.1"
    DEBUG: bool = False  # 发布版本关闭调试
    
    # 版本更新配置
    UPDATE_CHECK_URL: str = "https://your-update-server.com/api/version"
    DOWNLOAD_PAGE_URL: str = "https://your-website.com/download"
    FORCE_UPDATE_ENABLED: bool = True
    
    # 服务器配置
    HOST: str = "0.0.0.0"
    PORT: int = 8800
    
    # 数据库
    DATABASE_URL: str = _DATABASE_URL
    
    # 安全配置
    SECRET_KEY: str = secrets.token_urlsafe(32)
    ENCRYPTION_KEY: str = _ENCRYPTION_KEY
    ACCESS_TOKEN_EXPIRE_MINUTES: int = 60 * 24 * 7  # 7天
    
    # 管理员密码（首次启动后建议修改）
    ADMIN_PASSWORD: str = "admin123"
    
    # 超级管理员密钥
    SUPER_ADMIN_KEY: str = "YOUR_SECRET_ADMIN_KEY_2024"
    
    # 订阅链接配置
    SUBSCRIBE_BASE_URL: str = ""
    
    # 监控配置
    MONITOR_INTERVAL: int = 300  # 5分钟
    
    # 项目路径（使用模块级变量）
    BASE_DIR: Path = _BASE_DIR
    DATA_DIR: Path = _DATA_DIR
    SCRIPTS_DIR: Path = _SCRIPTS_DIR
    
    class Config:
        env_file = ".env"
        env_file_encoding = "utf-8"


settings = Settings()

