# -*- coding: utf-8 -*-
"""
🔐 许可证验证模块 - 客户端使用（只能验证，不能生成卡密）

功能:
- 验证许可证有效性
- 机器码生成与绑定
- 本地加密存储
- 防时间回调
- 多重防破解保护

安全说明:
- 此模块只有验证功能
- 生成卡密需要管理员使用独立的 license_generator.py
"""

import os
import sys
import json
import hashlib
import base64
import uuid
import time
import hmac
import subprocess
import platform
from datetime import datetime, timedelta
from typing import Optional, Dict, Tuple
from dataclasses import dataclass
from enum import Enum

# Windows 注册表支持
if platform.system() == "Windows":
    try:
        import winreg
        REGISTRY_AVAILABLE = True
    except ImportError:
        REGISTRY_AVAILABLE = False
else:
    REGISTRY_AVAILABLE = False


def get_app_dir():
    """获取应用程序所在目录"""
    if getattr(sys, 'frozen', False):
        return os.path.dirname(sys.executable)
    else:
        # 源码运行，返回 backend 目录
        return os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))


class LicenseType(Enum):
    """许可证类型"""
    TRIAL = "trial"              # 试用版
    STARTER_MONTHLY = "starter_monthly"      # 创业版-月付
    STARTER_YEARLY = "starter_yearly"        # 创业版-年付
    MATRIX_MONTHLY = "matrix_monthly"        # 矩阵版-月付
    MATRIX_YEARLY = "matrix_yearly"          # 矩阵版-年付
    AGENCY_MONTHLY = "agency_monthly"        # 机构版-月付
    AGENCY_YEARLY = "agency_yearly"          # 机构版-年付
    # 【兼容旧版本】保留旧类型支持
    MONTHLY = "monthly"          # 旧版月付（映射到矩阵版月付）
    YEARLY = "yearly"            # 旧版年付（映射到矩阵版年付）
    LIFETIME = "lifetime"        # 旧版永久（映射到机构版年付）


# 旧类型到新类型的映射（用于功能限制）
OLD_TYPE_MAPPING = {
    LicenseType.MONTHLY: LicenseType.MATRIX_MONTHLY,
    LicenseType.YEARLY: LicenseType.MATRIX_YEARLY,
    LicenseType.LIFETIME: LicenseType.AGENCY_YEARLY,
}


@dataclass
class LicenseInfo:
    """许可证信息"""
    license_type: LicenseType
    license_key: str
    machine_code: str
    activated_date: str
    expire_date: Optional[str]
    days_remaining: int
    user_name: str
    is_valid: bool
    is_bound: bool
    message: str = ""


class LicenseVerifier:
    """许可证验证器 - 只能验证，不能生成"""
    
    # 版本功能限制 - 按生产力订阅模式
    VERSION_LIMITS = {
        # 试用版 - 免费体验
        LicenseType.TRIAL: {
            "days": 3,              # 试用期3天
            "name": "试用版",
            "max_nodes": 3,         # 最多3个节点
            "max_daily_nodes": 1,   # 每天只能创建1个节点
            "price": 0
        },
        
        # 🥉 创业版 (Starter) - 个人起号，小打小闹
        LicenseType.STARTER_MONTHLY: {
            "days": 30,
            "name": "🥉 创业版（月付）",
            "max_nodes": 10,        # 限制10个节点
            "max_daily_nodes": -1,  # 每日无限制
            "price": 49
        },
        LicenseType.STARTER_YEARLY: {
            "days": 365,
            "name": "🥉 创业版（年付）",
            "max_nodes": 10,
            "max_daily_nodes": -1,
            "price": 399
        },
        
        # 🥇 矩阵版 (Matrix) - 标准工作室，批量起号
        LicenseType.MATRIX_MONTHLY: {
            "days": 30,
            "name": "🥇 矩阵版（月付）",
            "max_nodes": 50,        # 限制50个节点
            "max_daily_nodes": -1,
            "price": 149
        },
        LicenseType.MATRIX_YEARLY: {
            "days": 365,
            "name": "🥇 矩阵版（年付）",
            "max_nodes": 50,
            "max_daily_nodes": -1,
            "price": 1299
        },
        
        # 💎 机构版 (Agency) - 大型MCN，无限铺号
        LicenseType.AGENCY_MONTHLY: {
            "days": 30,
            "name": "💎 机构版（月付）",
            "max_nodes": -1,        # 无限节点
            "max_daily_nodes": -1,
            "price": 399
        },
        LicenseType.AGENCY_YEARLY: {
            "days": 365,
            "name": "💎 机构版（年付）",
            "max_nodes": -1,
            "max_daily_nodes": -1,
            "price": 3999
        },
        
        # 【兼容旧版本】旧许可证类型（功能等同矩阵版/机构版）
        LicenseType.MONTHLY: {
            "days": 30,
            "name": "月付版（旧版）",
            "max_nodes": 50,        # 等同矩阵版
            "max_daily_nodes": -1,
            "price": 149
        },
        LicenseType.YEARLY: {
            "days": 365,
            "name": "年付版（旧版）",
            "max_nodes": 50,        # 等同矩阵版
            "max_daily_nodes": -1,
            "price": 1299
        },
        LicenseType.LIFETIME: {
            "days": None,           # 永久有效
            "name": "永久版（旧版）",
            "max_nodes": -1,        # 无限节点
            "max_daily_nodes": -1,
            "price": 9999
        }
    }
    
    # 加密密钥（必须与生成器一致！）
    _SECRET_KEY_1 = "NodeAutoTool@2026#Primary!Key"
    _SECRET_KEY_2 = "xK9$mN2#pL5@vQ8^NodeTool"
    _SECRET_KEY_3 = "全球节点自动化2026"
    _SALT = b"NodeAutoToolSalt2026"
    
    # Windows 注册表路径（深藏在系统关键路径下，不易被发现）
    _REGISTRY_PATH = r"SOFTWARE\Microsoft\Windows\CurrentVersion\AppModel\StateRepository\Cache"
    _REGISTRY_KEY_NAME = "InstallTime"  # 伪装成系统键名
    
    def __init__(self):
        self.base_dir = get_app_dir()
        self.license_file = os.path.join(self.base_dir, ".license")
        self.timestamp_file = os.path.join(self.base_dir, ".ts")
        # 【防破解】多重隐藏标记文件
        self.activation_mark_file = os.path.join(self.base_dir, ".am")  # activation mark
        self.device_history_file = os.path.join(self.base_dir, ".dh")  # device history
        self.usage_file = os.path.join(self.base_dir, ".usage")  # 使用统计
        self._license_info: Optional[LicenseInfo] = None
        self._time_tampered = False
        self._daily_node_count = 0  # 今日创建节点数
        self._last_date = ""  # 最后记录日期
        
        # 启动时检查时间完整性
        self._check_time_integrity()
        # 【防破解】记录设备历史
        self._record_device_history()
        # 【终极防护】检查注册表标记
        self._check_registry_mark()
        # 【商业化】加载使用统计
        self._load_usage()
    
    def get_machine_code(self) -> str:
        """生成机器码（基于硬件信息）"""
        try:
            identifiers = []
            
            if platform.system() == "Windows":
                # CPU ID
                try:
                    result = subprocess.run(
                        ['wmic', 'cpu', 'get', 'ProcessorId'],
                        capture_output=True, text=True,
                        creationflags=subprocess.CREATE_NO_WINDOW
                    )
                    cpu_id = result.stdout.strip().split('\n')[-1].strip()
                    if cpu_id and cpu_id != "ProcessorId":
                        identifiers.append(f"CPU:{cpu_id}")
                except:
                    pass
                
                # 主板序列号
                try:
                    result = subprocess.run(
                        ['wmic', 'baseboard', 'get', 'SerialNumber'],
                        capture_output=True, text=True,
                        creationflags=subprocess.CREATE_NO_WINDOW
                    )
                    board_id = result.stdout.strip().split('\n')[-1].strip()
                    if board_id and board_id not in ["To be filled by O.E.M.", "SerialNumber", ""]:
                        identifiers.append(f"MB:{board_id}")
                except:
                    pass
                
                # BIOS序列号
                try:
                    result = subprocess.run(
                        ['wmic', 'bios', 'get', 'SerialNumber'],
                        capture_output=True, text=True,
                        creationflags=subprocess.CREATE_NO_WINDOW
                    )
                    bios_id = result.stdout.strip().split('\n')[-1].strip()
                    if bios_id and bios_id not in ["To be filled by O.E.M.", "SerialNumber", ""]:
                        identifiers.append(f"BIOS:{bios_id}")
                except:
                    pass
                
                # 硬盘序列号
                try:
                    result = subprocess.run(
                        ['wmic', 'diskdrive', 'get', 'SerialNumber'],
                        capture_output=True, text=True,
                        creationflags=subprocess.CREATE_NO_WINDOW
                    )
                    lines = [l.strip() for l in result.stdout.strip().split('\n')[1:] if l.strip()]
                    if lines:
                        identifiers.append(f"DISK:{lines[0]}")
                except:
                    pass
            
            else:
                # Linux/Mac: 使用 MAC 地址和主机名
                identifiers.append(f"MAC:{uuid.getnode()}")
                identifiers.append(f"HOST:{platform.node()}")
            
            # MAC地址作为备用
            if len(identifiers) < 2:
                identifiers.append(f"MAC:{uuid.getnode()}")
            
            # 多重哈希生成机器码
            combined = "|".join(sorted(identifiers))
            
            hash1 = hashlib.sha256((combined + self._SECRET_KEY_1).encode()).hexdigest()
            hash2 = hashlib.sha512((hash1 + self._SECRET_KEY_2).encode()).hexdigest()
            signature = hmac.new(self._SALT, hash2.encode(), hashlib.sha256).hexdigest()
            
            machine_code = signature[:16].upper()
            return "-".join([machine_code[i:i+4] for i in range(0, 16, 4)])
            
        except Exception:
            fallback = hashlib.sha256(
                (str(uuid.getnode()) + self._SECRET_KEY_3).encode()
            ).hexdigest()[:16].upper()
            return "-".join([fallback[i:i+4] for i in range(0, 16, 4)])
    
    def activate(self, license_key: str) -> Tuple[bool, str]:
        """
        激活许可证
        
        Args:
            license_key: 许可证密钥
        
        Returns:
            (是否成功, 消息)
        """
        try:
            # 解析许可证
            if not license_key or len(license_key) < 10:
                return False, "许可证密钥格式无效"
            
            # 移除前缀和分隔符（支持新格式.和旧格式-）
            if license_key[1] == ".":
                # 新格式：使用.作为分隔符
                key_body = license_key[2:].replace(".", "")
            elif license_key[1] == "-":
                # 旧格式：使用-作为分隔符（可能有问题，但保持兼容）
                key_body = license_key[2:].replace("-", "")
            else:
                key_body = license_key.replace("-", "").replace(".", "")
            
            # 补齐base64填充
            padding = 4 - len(key_body) % 4
            if padding != 4:
                key_body += "=" * padding
            
            # 解码
            try:
                decoded = base64.urlsafe_b64decode(key_body).decode()
            except:
                return False, "许可证密钥无法解码，请检查是否输入正确"
            
            if "|" not in decoded:
                return False, "许可证格式错误"
            
            data_str, signature = decoded.rsplit("|", 1)
            
            # 验证签名
            sig1 = hashlib.sha256((data_str + self._SECRET_KEY_1).encode()).hexdigest()[:8]
            sig2 = hmac.new(self._SALT, sig1.encode(), hashlib.md5).hexdigest()[:4]
            expected_sig = sig1 + sig2
            
            if signature != expected_sig:
                return False, "许可证签名无效，可能是盗版密钥"
            
            data = json.loads(data_str)
            license_type_str = data.get("type", "")
            
            # 【防破解】试用版激活限制 - 只有试用版需要检查激活历史
            # 付费版可以随时续费激活，不受此限制
            if license_type_str == "trial":
                if self._check_activation_history():
                    return False, "此设备已使用过试用版，请购买正式版许可证"
            
            # 检查激活窗口期（创建后180天内必须激活）
            created = datetime.strptime(data["created"][:10], "%Y-%m-%d")
            activation_window = created + timedelta(days=180)
            if datetime.now() > activation_window:
                return False, "许可证密钥已超过激活期限（180天），请联系客服"
            
            # 获取当前机器码
            current_mc = self.get_machine_code()
            
            # 检查是否已绑定其他设备（防止一码多用）
            if data.get("bound") and data.get("mc"):
                if data["mc"] != current_mc:
                    return False, "此许可证已绑定其他设备，无法在本机使用"
            
            # 首次激活，绑定设备
            data["bound"] = True
            data["mc"] = current_mc
            data["activated"] = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            
            # 保存到本地
            self._save_license(data, license_key)
            
            # 【防破解】标记设备已激活（不可删除的标记）
            self._mark_activation(current_mc, license_key)
            # 【终极防护】写入注册表标记（系统级防护）
            self._mark_registry(current_mc, license_key)
            
            # 重置缓存
            self._license_info = None
            
            license_type = LicenseType(data["type"])
            type_name = self.VERSION_LIMITS[license_type]["name"]
            days = data.get("days")
            
            if days:
                return True, f"🎉 激活成功！\n类型: {type_name}\n有效期: {days}天\n设备已绑定"
            else:
                return True, f"🎉 激活成功！\n类型: {type_name}\n有效期: 永久\n设备已绑定"
            
        except json.JSONDecodeError:
            return False, "许可证数据损坏"
        except Exception as e:
            return False, f"激活失败: {str(e)}"
    
    def verify(self) -> LicenseInfo:
        """验证当前许可证状态"""
        if self._license_info:
            return self._license_info
        
        current_mc = self.get_machine_code()
        
        # 默认未授权状态
        default_info = LicenseInfo(
            license_type=LicenseType.TRIAL,
            license_key="",
            machine_code=current_mc,
            activated_date="",
            expire_date=None,
            days_remaining=-1,
            user_name="未授权",
            is_valid=False,
            is_bound=False,
            message="请激活软件"
        )
        
        # 检测时间篡改
        if self._time_tampered:
            default_info.message = "检测到系统时间异常，请恢复正确时间"
            self._license_info = default_info
            return default_info
        
        if not os.path.exists(self.license_file):
            # 【防破解】检查是否有激活历史（防止删除文件重新试用）
            if self._check_activation_history():
                default_info.message = "检测到许可证文件被删除，请重新安装软件或联系客服"
                default_info.user_name = "文件异常"
                self._license_info = default_info
                return default_info
            
            # 【自动试用】首次运行，自动启动试用模式
            trial_info = self._start_auto_trial()
            if trial_info:
                self._license_info = trial_info
                return trial_info
            
            self._license_info = default_info
            return default_info
        
        try:
            with open(self.license_file, 'r', encoding='utf-8') as f:
                encrypted = f.read()
            
            decrypted = self._decrypt_data(encrypted)
            data = json.loads(decrypted)
            
            # 验证校验和
            stored_checksum = data.get("_checksum", "")
            calculated_checksum = self._calculate_checksum(data)
            if stored_checksum != calculated_checksum:
                default_info.message = "许可证数据被篡改"
                self._license_info = default_info
                return default_info
            
            # 验证机器码
            current_mc = self.get_machine_code()
            if data.get("mc") and data["mc"] != current_mc:
                default_info.message = "设备不匹配，请重新激活"
                self._license_info = default_info
                return default_info
            
            license_type = LicenseType(data["type"])
            limits = self.VERSION_LIMITS[license_type]
            
            # 计算有效期
            expire_date = None
            days_remaining = -1
            is_valid = True
            validity_days = limits.get("days")
            
            if validity_days is not None:
                activated_str = data.get("activated", "")
                if activated_str:
                    start_date = datetime.strptime(activated_str[:10], "%Y-%m-%d")
                else:
                    start_date = datetime.strptime(data["created"][:10], "%Y-%m-%d")
                
                expire = start_date + timedelta(days=validity_days)
                expire_date = expire.strftime("%Y-%m-%d")
                days_remaining = (expire - datetime.now()).days
                
                if datetime.now() > expire:
                    is_valid = False
                    days_remaining = 0
            else:
                days_remaining = 99999  # 永久
            
            message = "许可证有效" if is_valid else "许可证已过期，请续费"
            
            self._license_info = LicenseInfo(
                license_type=license_type,
                license_key=data.get("_key", ""),
                machine_code=data.get("mc", ""),
                activated_date=data.get("activated", ""),
                expire_date=expire_date,
                days_remaining=days_remaining,
                user_name=data.get("user", ""),
                is_valid=is_valid,
                is_bound=data.get("bound", False),
                message=message
            )
            
            return self._license_info
            
        except Exception as e:
            default_info.message = f"验证失败: {str(e)}"
            self._license_info = default_info
            return default_info
    
    def _start_auto_trial(self) -> Optional[LicenseInfo]:
        """
        【自动试用】首次运行时自动启动试用模式
        
        - 无需输入激活码
        - 自动绑定设备
        - 3天有效期
        - 功能受限（节点数量限制）
        """
        try:
            current_mc = self.get_machine_code()
            limits = self.VERSION_LIMITS[LicenseType.TRIAL]
            
            # 构建试用许可证数据
            trial_data = {
                "id": f"TRIAL-{current_mc[:8]}",
                "type": LicenseType.TRIAL.value,
                "user": "试用用户",
                "created": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                "activated": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                "days": limits["days"],
                "bound": True,
                "mc": current_mc,
                "auto_trial": True,  # 标记为自动试用
                "v": "1.0"
            }
            
            # 保存试用许可证
            self._save_license(trial_data, f"AUTO-TRIAL-{current_mc[:16]}")
            
            # 【安全】标记设备已激活（防止删除后重新试用）
            self._mark_activation(current_mc, f"AUTO-TRIAL-{current_mc[:16]}")
            # 【安全】写入注册表标记（双重防护）
            self._mark_registry(current_mc, f"AUTO-TRIAL-{current_mc[:16]}")
            
            # 计算到期时间
            start_date = datetime.now()
            expire = start_date + timedelta(days=limits["days"])
            expire_date = expire.strftime("%Y-%m-%d")
            days_remaining = limits["days"]
            
            return LicenseInfo(
                license_type=LicenseType.TRIAL,
                license_key=f"AUTO-TRIAL-{current_mc[:16]}",
                machine_code=current_mc,
                activated_date=trial_data["activated"],
                expire_date=expire_date,
                days_remaining=days_remaining,
                user_name="试用用户",
                is_valid=True,  # 试用期内有效
                is_bound=True,
                message=f"🎉 试用已激活！有效期{days_remaining}天，限{limits['max_nodes']}个节点"
            )
            
        except Exception as e:
            print(f"启动自动试用失败: {e}")
            return None
    
    def is_activated(self) -> bool:
        """检查是否已激活"""
        info = self.verify()
        return info.is_valid and info.is_bound
    
    def get_type_name(self) -> str:
        """获取许可证类型名称"""
        info = self.verify()
        return self.VERSION_LIMITS[info.license_type]["name"]
    
    def _save_license(self, data: dict, original_key: str):
        """加密保存许可证"""
        data["_key"] = original_key
        data["_checksum"] = self._calculate_checksum(data)
        data["_timestamp"] = int(time.time())
        
        json_str = json.dumps(data, ensure_ascii=False)
        encrypted = self._encrypt_data(json_str)
        
        with open(self.license_file, 'w', encoding='utf-8') as f:
            f.write(encrypted)
    
    def _calculate_checksum(self, data: dict) -> str:
        """计算数据校验和"""
        check_data = {k: v for k, v in data.items() if not k.startswith("_")}
        data_str = json.dumps(check_data, sort_keys=True, ensure_ascii=False)
        return hmac.new(
            self._SALT,
            (data_str + self._SECRET_KEY_2).encode(),
            hashlib.sha256
        ).hexdigest()[:16]
    
    def _encrypt_data(self, data: str) -> str:
        """加密数据"""
        data_bytes = data.encode('utf-8')
        b64_data = base64.b64encode(data_bytes).decode('ascii')
        
        key = self._SECRET_KEY_1
        encrypted_chars = []
        for i, char in enumerate(b64_data):
            encrypted_char = chr(ord(char) ^ ord(key[i % len(key)]))
            encrypted_chars.append(encrypted_char)
        encrypted = ''.join(encrypted_chars)
        
        final = base64.b64encode(encrypted.encode('latin-1')).decode('ascii')
        checksum = hashlib.md5(final.encode()).hexdigest()[:8]
        return checksum + final
    
    def _decrypt_data(self, data: str) -> str:
        """解密数据"""
        if len(data) < 8:
            raise ValueError("数据损坏")
        
        checksum = data[:8]
        encrypted = data[8:]
        
        if hashlib.md5(encrypted.encode()).hexdigest()[:8] != checksum:
            raise ValueError("数据校验失败")
        
        encrypted_bytes = base64.b64decode(encrypted).decode('latin-1')
        
        key = self._SECRET_KEY_1
        decrypted_chars = []
        for i, char in enumerate(encrypted_bytes):
            decrypted_char = chr(ord(char) ^ ord(key[i % len(key)]))
            decrypted_chars.append(decrypted_char)
        b64_data = ''.join(decrypted_chars)
        
        original = base64.b64decode(b64_data).decode('utf-8')
        return original
    
    def _check_time_integrity(self):
        """检查时间完整性（防时间回调）"""
        current_ts = int(time.time())
        
        try:
            if os.path.exists(self.timestamp_file):
                with open(self.timestamp_file, 'r') as f:
                    content = f.read().strip()
                    decoded = self._simple_decrypt(content)
                    stored_ts = int(decoded)
                    
                    # 时间回调超过1小时视为异常
                    if current_ts < stored_ts - 3600:
                        self._time_tampered = True
        except:
            pass
        
        # 更新时间戳
        self._save_timestamp(current_ts)
    
    def _save_timestamp(self, ts: int):
        """保存时间戳"""
        try:
            encrypted = self._simple_encrypt(str(ts))
            with open(self.timestamp_file, 'w') as f:
                f.write(encrypted)
        except:
            pass
    
    # ========== 【防破解】多重防护机制 ==========
    
    def _mark_activation(self, machine_code: str, license_key: str):
        """
        【防破解】标记设备已激活（即使删除许可证文件也能检测到）
        """
        try:
            mark_data = {
                "mc": machine_code,
                "first_activation": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
                "key_hash": hashlib.sha256(license_key.encode()).hexdigest()[:16],
                "ts": int(time.time())
            }
            encrypted = self._simple_encrypt(json.dumps(mark_data))
            with open(self.activation_mark_file, 'w') as f:
                f.write(encrypted)
        except:
            pass
    
    def _check_activation_history(self) -> bool:
        """
        【防破解】检查设备是否有激活历史（防止删除文件后重新激活）
        - 检查本地标记文件
        - 检查注册表标记（双重验证）
        """
        # 检查本地文件标记
        try:
            if os.path.exists(self.activation_mark_file):
                with open(self.activation_mark_file, 'r') as f:
                    content = f.read().strip()
                    decrypted = self._simple_decrypt(content)
                    data = json.loads(decrypted)
                    
                    # 验证机器码
                    if data.get("mc") == self.get_machine_code():
                        # 如果许可证文件不存在，说明用户删除了
                        if not os.path.exists(self.license_file):
                            return True
        except:
            pass
        
        # 【终极防护】检查注册表标记
        if self._check_registry_mark():
            # 如果注册表有标记但许可证文件不存在，说明文件被删除
            if not os.path.exists(self.license_file):
                return True
        
        return False
    
    def _record_device_history(self):
        """
        【防破解】记录设备历史（用于审计和防作弊）
        """
        try:
            history = []
            if os.path.exists(self.device_history_file):
                with open(self.device_history_file, 'r') as f:
                    content = f.read().strip()
                    if content:
                        decrypted = self._simple_decrypt(content)
                        history = json.loads(decrypted)
            
            # 添加当前运行记录
            current_record = {
                "mc": self.get_machine_code(),
                "ts": int(time.time()),
                "date": datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            }
            
            # 保留最近100条记录
            history.append(current_record)
            history = history[-100:]
            
            encrypted = self._simple_encrypt(json.dumps(history))
            with open(self.device_history_file, 'w') as f:
                f.write(encrypted)
        except:
            pass
    
    # ========== 辅助加密方法 ==========
    
    def _simple_encrypt(self, data: str) -> str:
        """简单加密"""
        key = self._SECRET_KEY_2
        result = []
        for i, c in enumerate(data):
            result.append(chr(ord(c) ^ ord(key[i % len(key)])))
        return base64.b64encode(''.join(result).encode('latin-1')).decode()
    
    def _simple_decrypt(self, data: str) -> str:
        """简单解密"""
        key = self._SECRET_KEY_2
        decoded = base64.b64decode(data).decode('latin-1')
        result = []
        for i, c in enumerate(decoded):
            result.append(chr(ord(c) ^ ord(key[i % len(key)])))
        return ''.join(result)
    
    # ========== 【终极防护】Windows 注册表标记 ==========
    
    def _mark_registry(self, machine_code: str, license_key: str):
        """
        【终极防护】在 Windows 注册表中写入激活标记
        - 即使删除所有文件，注册表记录仍然存在
        - 伪装成系统键名，不易被发现
        - 需要管理员权限才能删除
        """
        if not REGISTRY_AVAILABLE:
            return
        
        try:
            # 创建加密标记数据
            mark_data = {
                "mc": machine_code,
                "ts": int(time.time()),
                "kh": hashlib.sha256(license_key.encode()).hexdigest()[:12]
            }
            encrypted_value = self._simple_encrypt(json.dumps(mark_data))
            
            # 打开或创建注册表键
            try:
                key = winreg.OpenKey(
                    winreg.HKEY_CURRENT_USER,
                    self._REGISTRY_PATH,
                    0,
                    winreg.KEY_WRITE
                )
            except FileNotFoundError:
                # 如果路径不存在，创建它
                key = winreg.CreateKey(
                    winreg.HKEY_CURRENT_USER,
                    self._REGISTRY_PATH
                )
            
            # 写入加密数据（伪装成时间戳）
            winreg.SetValueEx(
                key,
                self._REGISTRY_KEY_NAME,
                0,
                winreg.REG_SZ,
                encrypted_value
            )
            winreg.CloseKey(key)
            
        except Exception as e:
            # 静默失败，不影响主功能
            pass
    
    def _check_registry_mark(self) -> bool:
        """
        【终极防护】检查注册表中的激活标记
        - 即使所有文件被删除，这个检查仍然有效
        """
        if not REGISTRY_AVAILABLE:
            return False
        
        try:
            key = winreg.OpenKey(
                winreg.HKEY_CURRENT_USER,
                self._REGISTRY_PATH,
                0,
                winreg.KEY_READ
            )
            
            try:
                value, _ = winreg.QueryValueEx(key, self._REGISTRY_KEY_NAME)
                winreg.CloseKey(key)
                
                # 解密并验证
                decrypted = self._simple_decrypt(value)
                data = json.loads(decrypted)
                
                # 验证机器码
                if data.get("mc") == self.get_machine_code():
                    return True
                    
            except FileNotFoundError:
                winreg.CloseKey(key)
                return False
                
        except Exception:
            return False
        
        return False
    
    # ========== 【商业化】使用限制和统计 ==========
    
    def _load_usage(self):
        """加载使用统计"""
        try:
            if os.path.exists(self.usage_file):
                with open(self.usage_file, 'r') as f:
                    content = f.read()
                    decoded = base64.b64decode(content).decode()
                    data = json.loads(decoded)
                    self._daily_node_count = data.get("count", 0)
                    self._last_date = data.get("date", "")
        except:
            pass
    
    def _save_usage(self):
        """保存使用统计"""
        try:
            data = {
                "count": self._daily_node_count,
                "date": self._last_date,
                "ts": int(time.time())
            }
            encoded = base64.b64encode(json.dumps(data).encode()).decode()
            with open(self.usage_file, 'w') as f:
                f.write(encoded)
        except:
            pass
    
    def can_create_node(self, current_node_count: int = 0) -> Tuple[bool, str]:
        """
        检查是否可以创建节点（商业化限制）
        
        Args:
            current_node_count: 当前数据库中的节点总数（由调用方传入）
        
        Returns:
            (是否可以, 消息)
        """
        info = self.verify()
        
        # 未激活
        if not info.is_valid:
            return False, "请先激活软件"
        
        # 获取限制
        limits = self.VERSION_LIMITS[info.license_type]
        max_nodes = limits.get("max_nodes", -1)
        max_daily = limits.get("max_daily_nodes", -1)
        
        # 无限制版本
        if max_nodes == -1 and max_daily == -1:
            return True, "OK"
        
        # 【修复】检查总节点数限制
        if max_nodes != -1 and current_node_count >= max_nodes:
            return False, f"试用版最多只能创建{max_nodes}个节点，当前已有{current_node_count}个，请升级许可证"
        
        # 检查今日创建数量
        if max_daily != -1:
            today = datetime.now().strftime("%Y-%m-%d")
            if self._last_date != today:
                self._daily_node_count = 0
                self._last_date = today
                self._save_usage()
            
            if self._daily_node_count >= max_daily:
                return False, f"试用版每天只能创建{max_daily}个节点，请升级许可证"
        
        return True, "OK"
    
    def record_node_creation(self):
        """记录节点创建（试用版限制）"""
        today = datetime.now().strftime("%Y-%m-%d")
        if self._last_date != today:
            self._daily_node_count = 0
            self._last_date = today
        
        self._daily_node_count += 1
        self._save_usage()
    
    def get_daily_remaining(self) -> int:
        """获取今日剩余可创建节点数"""
        info = self.verify()
        limits = self.VERSION_LIMITS[info.license_type]
        max_daily = limits.get("max_daily_nodes", -1)
        
        if max_daily == -1:
            return -1  # 无限制
        
        today = datetime.now().strftime("%Y-%m-%d")
        if self._last_date != today:
            return max_daily
        
        return max(0, max_daily - self._daily_node_count)
    
    def get_expiry_warning(self) -> Optional[Dict]:
        """
        获取到期提醒
        
        Returns:
            None 或 {"days_left": int, "message": str}
        """
        info = self.verify()
        
        if not info.is_valid:
            return None
        
        # 永久版不提醒
        if info.days_remaining > 9999:
            return None
        
        # 3天内提醒
        if info.days_remaining <= 3 and info.days_remaining > 0:
            return {
                "days_left": info.days_remaining,
                "message": f"⚠️ 您的许可证还有{info.days_remaining}天到期，请及时续费！",
                "type": "warning"
            }
        
        # 已过期
        if info.days_remaining <= 0:
            return {
                "days_left": 0,
                "message": "❌ 您的许可证已过期，节点功能已被限制，请立即续费！",
                "type": "error"
            }
        
        return None


# 全局验证器实例
_verifier: Optional[LicenseVerifier] = None

def get_license_verifier() -> LicenseVerifier:
    """获取许可证验证器单例"""
    global _verifier
    if _verifier is None:
        _verifier = LicenseVerifier()
    return _verifier
