"""
日志管理模块 - 提供完整的日志记录功能
"""
import os
import sys
import logging
import traceback
from datetime import datetime
from pathlib import Path
from logging.handlers import RotatingFileHandler, TimedRotatingFileHandler
from typing import Optional
import json


# ============== 日志目录 ==============
def get_log_dir() -> Path:
    """获取日志目录"""
    # 优先使用环境变量指定的目录
    if os.getenv("LOG_DIR"):
        log_dir = Path(os.getenv("LOG_DIR"))
    else:
        # 默认在应用根目录下的 logs 文件夹
        log_dir = Path(__file__).parent.parent.parent / "logs"
    
    log_dir.mkdir(parents=True, exist_ok=True)
    return log_dir


# ============== 自定义格式化器 ==============
class ColoredFormatter(logging.Formatter):
    """带颜色的控制台输出格式化器"""
    
    COLORS = {
        'DEBUG': '\033[36m',     # 青色
        'INFO': '\033[32m',      # 绿色
        'WARNING': '\033[33m',   # 黄色
        'ERROR': '\033[31m',     # 红色
        'CRITICAL': '\033[41m',  # 红底
    }
    RESET = '\033[0m'
    
    def format(self, record):
        color = self.COLORS.get(record.levelname, self.RESET)
        record.levelname = f"{color}{record.levelname}{self.RESET}"
        return super().format(record)


class JSONFormatter(logging.Formatter):
    """JSON格式化器 - 便于日志分析"""
    
    def format(self, record):
        log_data = {
            "timestamp": datetime.fromtimestamp(record.created).isoformat(),
            "level": record.levelname,
            "logger": record.name,
            "message": record.getMessage(),
            "module": record.module,
            "function": record.funcName,
            "line": record.lineno,
        }
        
        # 添加异常信息
        if record.exc_info:
            log_data["exception"] = {
                "type": record.exc_info[0].__name__ if record.exc_info[0] else None,
                "message": str(record.exc_info[1]) if record.exc_info[1] else None,
                "traceback": traceback.format_exception(*record.exc_info)
            }
        
        # 添加额外数据
        if hasattr(record, 'extra_data'):
            log_data["extra"] = record.extra_data
        
        return json.dumps(log_data, ensure_ascii=False, default=str)


# ============== 日志过滤器 ==============
class ErrorOnlyFilter(logging.Filter):
    """只允许ERROR及以上级别的日志"""
    def filter(self, record):
        return record.levelno >= logging.ERROR


# ============== 日志管理器 ==============
class LoggerManager:
    """日志管理器"""
    
    _instance: Optional['LoggerManager'] = None
    _initialized: bool = False
    
    def __new__(cls):
        if cls._instance is None:
            cls._instance = super().__new__(cls)
        return cls._instance
    
    def __init__(self):
        if self._initialized:
            return
        
        self.log_dir = get_log_dir()
        self._setup_logging()
        self._initialized = True
    
    def _setup_logging(self):
        """配置日志系统"""
        # 根日志器
        root_logger = logging.getLogger()
        root_logger.setLevel(logging.DEBUG)
        
        # 清除现有处理器
        root_logger.handlers.clear()
        
        # 1. 控制台处理器 - 带颜色
        console_handler = logging.StreamHandler(sys.stdout)
        console_handler.setLevel(logging.INFO)
        console_format = "%(asctime)s │ %(levelname)-8s │ %(name)s │ %(message)s"
        console_handler.setFormatter(ColoredFormatter(console_format, datefmt="%H:%M:%S"))
        root_logger.addHandler(console_handler)
        
        # 2. 主日志文件 - 按大小轮转 (10MB * 5个备份)
        main_log = self.log_dir / "app.log"
        file_handler = RotatingFileHandler(
            main_log,
            maxBytes=10 * 1024 * 1024,  # 10MB
            backupCount=5,
            encoding='utf-8'
        )
        file_handler.setLevel(logging.DEBUG)
        file_format = "%(asctime)s | %(levelname)-8s | %(name)s | %(funcName)s:%(lineno)d | %(message)s"
        file_handler.setFormatter(logging.Formatter(file_format))
        root_logger.addHandler(file_handler)
        
        # 3. 错误日志文件 - 按天轮转
        error_log = self.log_dir / "error.log"
        error_handler = TimedRotatingFileHandler(
            error_log,
            when='midnight',
            interval=1,
            backupCount=30,  # 保留30天
            encoding='utf-8'
        )
        error_handler.setLevel(logging.ERROR)
        error_handler.addFilter(ErrorOnlyFilter())
        error_handler.setFormatter(logging.Formatter(file_format))
        root_logger.addHandler(error_handler)
        
        # 4. JSON格式日志 - 便于分析
        json_log = self.log_dir / "app.json.log"
        json_handler = RotatingFileHandler(
            json_log,
            maxBytes=10 * 1024 * 1024,
            backupCount=3,
            encoding='utf-8'
        )
        json_handler.setLevel(logging.INFO)
        json_handler.setFormatter(JSONFormatter())
        root_logger.addHandler(json_handler)
        
        # 5. 操作审计日志 - 记录重要操作
        audit_log = self.log_dir / "audit.log"
        audit_handler = RotatingFileHandler(
            audit_log,
            maxBytes=5 * 1024 * 1024,
            backupCount=10,
            encoding='utf-8'
        )
        audit_handler.setLevel(logging.INFO)
        audit_handler.setFormatter(logging.Formatter(
            "%(asctime)s | %(levelname)s | AUDIT | %(message)s"
        ))
        # 审计日志使用专门的 logger
        audit_logger = logging.getLogger("audit")
        audit_logger.addHandler(audit_handler)
        audit_logger.propagate = False
    
    def get_logger(self, name: str = None) -> logging.Logger:
        """获取日志记录器"""
        return logging.getLogger(name)
    
    def get_audit_logger(self) -> logging.Logger:
        """获取审计日志记录器"""
        return logging.getLogger("audit")
    
    def get_log_files(self) -> list:
        """获取所有日志文件信息"""
        log_files = []
        for f in self.log_dir.glob("*.log"):
            stat = f.stat()
            log_files.append({
                "name": f.name,
                "path": str(f),
                "size": stat.st_size,
                "size_human": self._format_size(stat.st_size),
                "modified": datetime.fromtimestamp(stat.st_mtime).isoformat(),
            })
        return sorted(log_files, key=lambda x: x['name'])
    
    def read_log_file(self, filename: str, lines: int = 200, level: str = None) -> list:
        """读取日志文件内容"""
        log_file = self.log_dir / filename
        if not log_file.exists():
            return []
        
        result = []
        with open(log_file, 'r', encoding='utf-8', errors='ignore') as f:
            all_lines = f.readlines()
            
            # 过滤级别
            if level:
                all_lines = [l for l in all_lines if f"| {level}" in l]
            
            # 取最后N行
            for line in all_lines[-lines:]:
                result.append(line.strip())
        
        return result
    
    def clear_old_logs(self, days: int = 30):
        """清理旧日志"""
        from datetime import timedelta
        cutoff = datetime.now() - timedelta(days=days)
        
        count = 0
        for f in self.log_dir.glob("*.log*"):
            if datetime.fromtimestamp(f.stat().st_mtime) < cutoff:
                f.unlink()
                count += 1
        
        return count
    
    @staticmethod
    def _format_size(size: int) -> str:
        """格式化文件大小"""
        for unit in ['B', 'KB', 'MB', 'GB']:
            if size < 1024:
                return f"{size:.1f} {unit}"
            size /= 1024
        return f"{size:.1f} TB"


# ============== 便捷函数 ==============
_manager: Optional[LoggerManager] = None

def init_logging():
    """初始化日志系统"""
    global _manager
    _manager = LoggerManager()
    return _manager

def get_logger(name: str = None) -> logging.Logger:
    """获取日志记录器"""
    global _manager
    if _manager is None:
        _manager = LoggerManager()
    return _manager.get_logger(name)

def get_audit_logger() -> logging.Logger:
    """获取审计日志记录器"""
    global _manager
    if _manager is None:
        _manager = LoggerManager()
    return _manager.get_audit_logger()

def get_log_manager() -> LoggerManager:
    """获取日志管理器实例"""
    global _manager
    if _manager is None:
        _manager = LoggerManager()
    return _manager


# ============== 装饰器 ==============
def log_operation(operation: str):
    """记录操作的装饰器"""
    def decorator(func):
        async def async_wrapper(*args, **kwargs):
            logger = get_logger(func.__module__)
            audit = get_audit_logger()
            
            try:
                logger.debug(f"开始执行: {operation}")
                result = await func(*args, **kwargs)
                audit.info(f"✅ {operation} - 成功")
                return result
            except Exception as e:
                logger.error(f"执行失败: {operation} - {e}", exc_info=True)
                audit.error(f"❌ {operation} - 失败: {e}")
                raise
        
        def sync_wrapper(*args, **kwargs):
            logger = get_logger(func.__module__)
            audit = get_audit_logger()
            
            try:
                logger.debug(f"开始执行: {operation}")
                result = func(*args, **kwargs)
                audit.info(f"✅ {operation} - 成功")
                return result
            except Exception as e:
                logger.error(f"执行失败: {operation} - {e}", exc_info=True)
                audit.error(f"❌ {operation} - 失败: {e}")
                raise
        
        import asyncio
        if asyncio.iscoroutinefunction(func):
            return async_wrapper
        return sync_wrapper
    return decorator

