"""
安全工具：加密、解密、Token生成
"""
import base64
import hashlib
import secrets
from datetime import datetime, timedelta
from typing import Optional

from cryptography.fernet import Fernet
from jose import jwt
from passlib.context import CryptContext

from .config import settings


# 密码哈希
pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")


def get_fernet() -> Fernet:
    """获取Fernet加密实例"""
    # 将密钥转换为32字节的base64编码
    key = hashlib.sha256(settings.ENCRYPTION_KEY.encode()).digest()
    key_b64 = base64.urlsafe_b64encode(key)
    return Fernet(key_b64)


def encrypt_string(plain_text: str) -> str:
    """加密字符串"""
    if not plain_text:
        return ""
    f = get_fernet()
    return f.encrypt(plain_text.encode()).decode()


def decrypt_string(encrypted_text: str) -> str:
    """解密字符串"""
    if not encrypted_text:
        return ""
    try:
        f = get_fernet()
        return f.decrypt(encrypted_text.encode()).decode()
    except Exception:
        return ""


def hash_password(password: str) -> str:
    """密码哈希"""
    return pwd_context.hash(password)


def verify_password(plain_password: str, hashed_password: str) -> bool:
    """验证密码"""
    return pwd_context.verify(plain_password, hashed_password)


def create_access_token(data: dict, expires_delta: Optional[timedelta] = None) -> str:
    """创建访问令牌"""
    to_encode = data.copy()
    if expires_delta:
        expire = datetime.utcnow() + expires_delta
    else:
        expire = datetime.utcnow() + timedelta(minutes=settings.ACCESS_TOKEN_EXPIRE_MINUTES)
    to_encode.update({"exp": expire})
    encoded_jwt = jwt.encode(to_encode, settings.SECRET_KEY, algorithm="HS256")
    return encoded_jwt


def verify_token(token: str) -> Optional[dict]:
    """验证令牌"""
    try:
        payload = jwt.decode(token, settings.SECRET_KEY, algorithms=["HS256"])
        return payload
    except Exception:
        return None


def generate_uuid() -> str:
    """生成UUID (用于VLESS)"""
    import uuid
    return str(uuid.uuid4())


def generate_subscribe_token() -> str:
    """生成订阅令牌"""
    return secrets.token_urlsafe(16)


def generate_reality_keys() -> tuple[str, str]:
    """生成Reality密钥对 (私钥, 公钥)"""
    # 使用x25519生成密钥对
    from cryptography.hazmat.primitives.asymmetric import x25519
    from cryptography.hazmat.primitives import serialization
    
    private_key = x25519.X25519PrivateKey.generate()
    public_key = private_key.public_key()
    
    # 转换为base64格式
    private_bytes = private_key.private_bytes(
        encoding=serialization.Encoding.Raw,
        format=serialization.PrivateFormat.Raw,
        encryption_algorithm=serialization.NoEncryption()
    )
    public_bytes = public_key.public_bytes(
        encoding=serialization.Encoding.Raw,
        format=serialization.PublicFormat.Raw
    )
    
    private_key_b64 = base64.urlsafe_b64encode(private_bytes).decode().rstrip('=')
    public_key_b64 = base64.urlsafe_b64encode(public_bytes).decode().rstrip('=')
    
    return private_key_b64, public_key_b64


def generate_short_id() -> str:
    """生成Reality Short ID"""
    return secrets.token_hex(8)




