"""
客户模型
"""
from datetime import datetime
from typing import Optional, List, TYPE_CHECKING
from sqlalchemy import String, Text, DateTime, Enum as SQLEnum
from sqlalchemy.orm import Mapped, mapped_column, relationship
import enum

from ..core.database import Base

if TYPE_CHECKING:
    from .server import Server
    from .socks5 import Socks5Proxy
    from .node import Node


class CustomerType(str, enum.Enum):
    """客户类型"""
    LIVE = "live"           # 直播
    SHORT_VIDEO = "short_video"  # 短视频
    OTHER = "other"         # 其他


class Customer(Base):
    """客户表"""
    __tablename__ = "customers"
    
    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    name: Mapped[str] = mapped_column(String(100), comment="客户名称")
    contact: Mapped[Optional[str]] = mapped_column(String(100), nullable=True, comment="联系方式")
    customer_type: Mapped[CustomerType] = mapped_column(
        SQLEnum(CustomerType), 
        default=CustomerType.LIVE,
        comment="客户类型"
    )
    expire_at: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True, comment="服务到期时间")
    remark: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="备注")
    
    # 客户级订阅令牌（一键订阅该客户所有节点）
    subscribe_token: Mapped[Optional[str]] = mapped_column(String(50), nullable=True, unique=True, comment="订阅令牌")
    
    created_at: Mapped[datetime] = mapped_column(DateTime, default=datetime.now, comment="创建时间")
    updated_at: Mapped[datetime] = mapped_column(
        DateTime, 
        default=datetime.now, 
        onupdate=datetime.now,
        comment="更新时间"
    )
    
    # 关联
    servers: Mapped[List["Server"]] = relationship(back_populates="customer", cascade="all, delete-orphan")
    socks5_proxies: Mapped[List["Socks5Proxy"]] = relationship(back_populates="customer", cascade="all, delete-orphan")
    nodes: Mapped[List["Node"]] = relationship(back_populates="customer", cascade="all, delete-orphan")
    
    def __repr__(self):
        return f"<Customer(id={self.id}, name={self.name})>"




