"""
域名模型
"""
from datetime import datetime
from typing import Optional, List, TYPE_CHECKING
from sqlalchemy import String, Text, DateTime, ForeignKey, Enum as SQLEnum
from sqlalchemy.orm import Mapped, mapped_column, relationship
import enum

from ..core.database import Base

if TYPE_CHECKING:
    from .customer import Customer
    from .node import Node


class SSLStatus(str, enum.Enum):
    """SSL证书状态"""
    NONE = "none"           # 无证书
    PENDING = "pending"     # 申请中
    VALID = "valid"         # 有效
    EXPIRED = "expired"     # 已过期
    ERROR = "error"         # 错误


class Domain(Base):
    """域名表"""
    __tablename__ = "domains"
    
    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    customer_id: Mapped[Optional[int]] = mapped_column(
        ForeignKey("customers.id", ondelete="SET NULL"), 
        nullable=True,
        comment="关联客户ID"
    )
    
    domain: Mapped[str] = mapped_column(String(255), unique=True, comment="域名")
    ssl_status: Mapped[SSLStatus] = mapped_column(
        SQLEnum(SSLStatus), 
        default=SSLStatus.NONE,
        comment="SSL状态"
    )
    ssl_cert_path: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="证书路径")
    ssl_key_path: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="私钥路径")
    ssl_expire_at: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True, comment="证书过期时间")
    
    remark: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="备注")
    created_at: Mapped[datetime] = mapped_column(DateTime, default=datetime.now, comment="创建时间")
    updated_at: Mapped[datetime] = mapped_column(
        DateTime, 
        default=datetime.now, 
        onupdate=datetime.now,
        comment="更新时间"
    )
    
    # 关联
    customer: Mapped[Optional["Customer"]] = relationship("Customer")
    nodes: Mapped[List["Node"]] = relationship(back_populates="domain")
    
    def __repr__(self):
        return f"<Domain(id={self.id}, domain={self.domain})>"




