"""
软件授权模型
"""
import enum
import secrets
from datetime import datetime
from typing import Optional, List, TYPE_CHECKING
from sqlalchemy import String, Text, Integer, DateTime, Enum as SQLEnum, Boolean, ForeignKey
from sqlalchemy.orm import Mapped, mapped_column, relationship

from ..core.database import Base


class LicenseType(str, enum.Enum):
    """授权类型"""
    TRIAL = "trial"           # 试用版 (7天)
    MONTHLY = "monthly"       # 月卡 (30天)
    QUARTERLY = "quarterly"   # 季卡 (90天)
    YEARLY = "yearly"         # 年卡 (365天)
    LIFETIME = "lifetime"     # 永久卡


class LicenseStatus(str, enum.Enum):
    """授权状态"""
    UNUSED = "unused"         # 未使用
    ACTIVE = "active"         # 已激活
    EXPIRED = "expired"       # 已过期
    DISABLED = "disabled"     # 已禁用


class License(Base):
    """软件授权表"""
    __tablename__ = "licenses"
    
    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    
    # 卡密信息
    code: Mapped[str] = mapped_column(String(50), unique=True, index=True, comment="授权码")
    license_type: Mapped[LicenseType] = mapped_column(
        SQLEnum(LicenseType), 
        default=LicenseType.MONTHLY,
        comment="授权类型"
    )
    
    # 有效期设置（天数，永久卡设为0或NULL）
    duration_days: Mapped[int] = mapped_column(Integer, default=30, comment="有效期天数")
    
    # 设备绑定限制
    max_devices: Mapped[int] = mapped_column(Integer, default=1, comment="最大绑定设备数")
    
    # 状态
    status: Mapped[LicenseStatus] = mapped_column(
        SQLEnum(LicenseStatus),
        default=LicenseStatus.UNUSED,
        comment="状态"
    )
    
    # 激活信息
    activated_at: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True, comment="激活时间")
    expire_at: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True, comment="到期时间")
    
    # 绑定设备信息（浏览器指纹）
    device_fingerprint: Mapped[Optional[str]] = mapped_column(String(100), nullable=True, comment="设备指纹")
    device_info: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="设备信息(JSON)")
    
    # 用户信息
    user_remark: Mapped[Optional[str]] = mapped_column(String(200), nullable=True, comment="用户备注")
    
    # 管理信息
    created_by: Mapped[Optional[str]] = mapped_column(String(50), nullable=True, comment="创建者")
    remark: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="管理备注")
    
    created_at: Mapped[datetime] = mapped_column(DateTime, default=datetime.now, comment="创建时间")
    updated_at: Mapped[datetime] = mapped_column(
        DateTime,
        default=datetime.now,
        onupdate=datetime.now,
        comment="更新时间"
    )
    
    @staticmethod
    def generate_code(prefix: str = "LIC") -> str:
        """生成授权码"""
        # 格式: LIC-XXXX-XXXX-XXXX
        part1 = secrets.token_hex(2).upper()
        part2 = secrets.token_hex(2).upper()
        part3 = secrets.token_hex(2).upper()
        return f"{prefix}-{part1}-{part2}-{part3}"
    
    @staticmethod
    def get_duration_days(license_type: LicenseType) -> int:
        """根据授权类型获取天数"""
        duration_map = {
            LicenseType.TRIAL: 7,
            LicenseType.MONTHLY: 30,
            LicenseType.QUARTERLY: 90,
            LicenseType.YEARLY: 365,
            LicenseType.LIFETIME: 36500,  # 100年，相当于永久
        }
        return duration_map.get(license_type, 30)
    
    def is_valid(self) -> bool:
        """检查授权是否有效"""
        if self.status != LicenseStatus.ACTIVE:
            return False
        if self.expire_at and datetime.now() > self.expire_at:
            return False
        return True
    
    def days_remaining(self) -> int:
        """剩余天数"""
        if not self.expire_at:
            return 36500  # 永久卡
        delta = self.expire_at - datetime.now()
        return max(0, delta.days)
    
    def __repr__(self):
        return f"<License(id={self.id}, code={self.code}, type={self.license_type}, status={self.status})>"


class SystemNotice(Base):
    """系统公告表"""
    __tablename__ = "system_notices"
    
    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    
    title: Mapped[str] = mapped_column(String(200), comment="公告标题")
    content: Mapped[str] = mapped_column(Text, comment="公告内容")
    notice_type: Mapped[str] = mapped_column(String(20), default="info", comment="类型: info/warning/danger")
    
    # 是否强制显示
    is_force: Mapped[bool] = mapped_column(Boolean, default=False, comment="是否强制显示")
    
    # 是否启用
    is_active: Mapped[bool] = mapped_column(Boolean, default=True, comment="是否启用")
    
    # 显示时间范围
    start_at: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True, comment="开始显示时间")
    end_at: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True, comment="结束显示时间")
    
    created_at: Mapped[datetime] = mapped_column(DateTime, default=datetime.now, comment="创建时间")
    
    def __repr__(self):
        return f"<SystemNotice(id={self.id}, title={self.title})>"


class SystemConfig(Base):
    """系统配置表"""
    __tablename__ = "system_config"
    
    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    
    key: Mapped[str] = mapped_column(String(50), unique=True, index=True, comment="配置键")
    value: Mapped[str] = mapped_column(Text, comment="配置值")
    description: Mapped[Optional[str]] = mapped_column(String(200), nullable=True, comment="描述")
    
    updated_at: Mapped[datetime] = mapped_column(
        DateTime,
        default=datetime.now,
        onupdate=datetime.now,
        comment="更新时间"
    )
    
    def __repr__(self):
        return f"<SystemConfig(key={self.key})>"

