"""
操作日志模型
"""
from datetime import datetime
from typing import Optional
from sqlalchemy import String, Text, Integer, DateTime, Enum as SQLEnum
from sqlalchemy.orm import Mapped, mapped_column
import enum

from ..core.database import Base


class LogAction(str, enum.Enum):
    """操作类型"""
    CREATE = "create"
    UPDATE = "update"
    DELETE = "delete"
    DEPLOY = "deploy"
    START = "start"
    STOP = "stop"
    TEST = "test"
    LOGIN = "login"


class LogTargetType(str, enum.Enum):
    """目标类型"""
    CUSTOMER = "customer"
    SERVER = "server"
    SOCKS5 = "socks5"
    DOMAIN = "domain"
    NODE = "node"
    SYSTEM = "system"


class OperationLog(Base):
    """操作日志表"""
    __tablename__ = "operation_logs"
    
    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    
    action: Mapped[LogAction] = mapped_column(SQLEnum(LogAction), comment="操作类型")
    target_type: Mapped[LogTargetType] = mapped_column(SQLEnum(LogTargetType), comment="目标类型")
    target_id: Mapped[Optional[int]] = mapped_column(Integer, nullable=True, comment="目标ID")
    target_name: Mapped[Optional[str]] = mapped_column(String(100), nullable=True, comment="目标名称")
    
    detail: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="详情")
    ip_address: Mapped[Optional[str]] = mapped_column(String(45), nullable=True, comment="操作IP")
    
    created_at: Mapped[datetime] = mapped_column(DateTime, default=datetime.now, comment="创建时间")
    
    def __repr__(self):
        return f"<OperationLog(id={self.id}, action={self.action}, target={self.target_type})>"




