"""
节点模型
"""
from datetime import datetime
from typing import Optional, TYPE_CHECKING
from sqlalchemy import String, Text, Integer, BigInteger, DateTime, ForeignKey, Enum as SQLEnum, JSON, Boolean
from sqlalchemy.orm import Mapped, mapped_column, relationship
import enum

from ..core.database import Base

if TYPE_CHECKING:
    from .customer import Customer
    from .server import Server
    from .socks5 import Socks5Proxy
    from .domain import Domain


class NodeProtocol(str, enum.Enum):
    """节点协议"""
    VLESS_REALITY = "vless-reality"     # VLESS + Reality
    VLESS_TLS_WS = "vless-tls-ws"       # VLESS + TLS + WebSocket


class NodeStatus(str, enum.Enum):
    """节点状态"""
    RUNNING = "running"     # 运行中
    STOPPED = "stopped"     # 已停止
    ERROR = "error"         # 错误
    DEPLOYING = "deploying" # 部署中
    UNKNOWN = "unknown"     # 未知


class Node(Base):
    """节点表"""
    __tablename__ = "nodes"
    
    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    
    # 关联
    customer_id: Mapped[int] = mapped_column(
        ForeignKey("customers.id", ondelete="CASCADE"),
        comment="所属客户"
    )
    server_id: Mapped[int] = mapped_column(
        ForeignKey("servers.id", ondelete="CASCADE"),
        comment="中转服务器"
    )
    socks5_id: Mapped[int] = mapped_column(
        ForeignKey("socks5_proxies.id", ondelete="CASCADE"),
        comment="SK5落地"
    )
    domain_id: Mapped[Optional[int]] = mapped_column(
        ForeignKey("domains.id", ondelete="SET NULL"),
        nullable=True,
        comment="域名（可选）"
    )
    
    # 基本信息
    name: Mapped[str] = mapped_column(String(100), comment="节点名称")
    protocol: Mapped[NodeProtocol] = mapped_column(
        SQLEnum(NodeProtocol), 
        default=NodeProtocol.VLESS_REALITY,
        comment="协议类型"
    )
    listen_port: Mapped[int] = mapped_column(Integer, default=443, comment="监听端口")
    uuid: Mapped[str] = mapped_column(String(50), comment="VLESS UUID")
    
    # Reality 配置
    reality_dest: Mapped[Optional[str]] = mapped_column(String(255), nullable=True, comment="Reality目标")
    reality_server_names: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="Reality SNI列表(JSON)")
    reality_private_key: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="Reality私钥")
    reality_public_key: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="Reality公钥")
    reality_short_id: Mapped[Optional[str]] = mapped_column(String(20), nullable=True, comment="Reality Short ID")
    
    # 完整配置
    config_json: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="Xray完整配置")
    
    # 分享链接
    share_link: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="VLESS分享链接")
    subscribe_token: Mapped[Optional[str]] = mapped_column(String(50), nullable=True, comment="订阅令牌")
    
    # 状态
    status: Mapped[NodeStatus] = mapped_column(
        SQLEnum(NodeStatus), 
        default=NodeStatus.UNKNOWN,
        comment="状态"
    )
    latency: Mapped[Optional[int]] = mapped_column(Integer, nullable=True, comment="延迟(ms)")
    last_check: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True, comment="最后检测时间")
    
    # 流量统计（单位：字节）
    traffic_up: Mapped[int] = mapped_column(BigInteger, default=0, comment="上行流量(字节)")
    traffic_down: Mapped[int] = mapped_column(BigInteger, default=0, comment="下行流量(字节)")
    traffic_total: Mapped[int] = mapped_column(BigInteger, default=0, comment="总流量(字节)")
    traffic_limit: Mapped[Optional[int]] = mapped_column(BigInteger, nullable=True, comment="流量限制(字节)")
    last_traffic_sync: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True, comment="最后流量同步时间")
    
    # 到期时间
    expire_at: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True, comment="到期时间")
    is_expired: Mapped[bool] = mapped_column(Boolean, default=False, comment="是否已过期")
    is_traffic_exceeded: Mapped[bool] = mapped_column(Boolean, default=False, comment="是否流量超限")
    
    # 时间
    created_at: Mapped[datetime] = mapped_column(DateTime, default=datetime.now, comment="创建时间")
    updated_at: Mapped[datetime] = mapped_column(
        DateTime, 
        default=datetime.now, 
        onupdate=datetime.now,
        comment="更新时间"
    )
    
    # 关联
    customer: Mapped["Customer"] = relationship(back_populates="nodes")
    server: Mapped["Server"] = relationship(back_populates="nodes")
    socks5: Mapped["Socks5Proxy"] = relationship(back_populates="nodes")
    domain: Mapped[Optional["Domain"]] = relationship(back_populates="nodes")
    
    def __repr__(self):
        return f"<Node(id={self.id}, name={self.name}, protocol={self.protocol})>"



