"""
服务器模型
"""
from datetime import datetime
from typing import Optional, List, TYPE_CHECKING
from sqlalchemy import String, Text, Integer, Boolean, DateTime, ForeignKey, Enum as SQLEnum
from sqlalchemy.orm import Mapped, mapped_column, relationship
import enum

from ..core.database import Base

if TYPE_CHECKING:
    from .customer import Customer
    from .node import Node


class ServerStatus(str, enum.Enum):
    """服务器状态"""
    ONLINE = "online"       # 在线
    OFFLINE = "offline"     # 离线
    UNKNOWN = "unknown"     # 未知
    ERROR = "error"         # 错误


class Server(Base):
    """服务器表（中转服务器）"""
    __tablename__ = "servers"
    
    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    customer_id: Mapped[Optional[int]] = mapped_column(
        ForeignKey("customers.id", ondelete="SET NULL"), 
        nullable=True,
        comment="关联客户ID（NULL表示自有服务器）"
    )
    
    name: Mapped[str] = mapped_column(String(100), comment="服务器名称")
    ip: Mapped[str] = mapped_column(String(45), comment="IP地址")
    ssh_port: Mapped[int] = mapped_column(Integer, default=22, comment="SSH端口")
    ssh_user: Mapped[str] = mapped_column(String(50), default="root", comment="SSH用户名")
    ssh_password: Mapped[str] = mapped_column(Text, comment="SSH密码（加密存储）")
    
    region: Mapped[Optional[str]] = mapped_column(String(50), nullable=True, comment="地区")
    status: Mapped[ServerStatus] = mapped_column(
        SQLEnum(ServerStatus), 
        default=ServerStatus.UNKNOWN,
        comment="状态"
    )
    is_own: Mapped[bool] = mapped_column(Boolean, default=False, comment="是否自有服务器")
    
    last_check: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True, comment="最后检测时间")
    created_at: Mapped[datetime] = mapped_column(DateTime, default=datetime.now, comment="创建时间")
    updated_at: Mapped[datetime] = mapped_column(
        DateTime, 
        default=datetime.now, 
        onupdate=datetime.now,
        comment="更新时间"
    )
    
    # 关联
    customer: Mapped[Optional["Customer"]] = relationship(back_populates="servers")
    nodes: Mapped[List["Node"]] = relationship(back_populates="server", cascade="all, delete-orphan")
    
    def __repr__(self):
        return f"<Server(id={self.id}, name={self.name}, ip={self.ip})>"




