"""
SOCKS5代理模型
"""
from datetime import datetime
from typing import Optional, List, TYPE_CHECKING
from sqlalchemy import String, Text, Integer, DateTime, ForeignKey, Enum as SQLEnum
from sqlalchemy.orm import Mapped, mapped_column, relationship
import enum

from ..core.database import Base

if TYPE_CHECKING:
    from .customer import Customer
    from .node import Node


class Socks5Status(str, enum.Enum):
    """SK5状态"""
    AVAILABLE = "available"     # 可用
    UNAVAILABLE = "unavailable" # 不可用
    UNKNOWN = "unknown"         # 未知


class Socks5Proxy(Base):
    """SOCKS5代理表（落地）"""
    __tablename__ = "socks5_proxies"
    
    id: Mapped[int] = mapped_column(primary_key=True, autoincrement=True)
    customer_id: Mapped[Optional[int]] = mapped_column(
        ForeignKey("customers.id", ondelete="SET NULL"), 
        nullable=True,
        comment="关联客户ID"
    )
    
    ip: Mapped[str] = mapped_column(String(45), comment="IP地址")
    port: Mapped[int] = mapped_column(Integer, comment="端口")
    username: Mapped[Optional[str]] = mapped_column(String(100), nullable=True, comment="用户名")
    password: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="密码（加密存储）")
    
    region: Mapped[Optional[str]] = mapped_column(String(100), nullable=True, comment="IP归属地")
    status: Mapped[Socks5Status] = mapped_column(
        SQLEnum(Socks5Status), 
        default=Socks5Status.UNKNOWN,
        comment="状态"
    )
    latency: Mapped[Optional[int]] = mapped_column(Integer, nullable=True, comment="延迟(ms)")
    
    last_check: Mapped[Optional[datetime]] = mapped_column(DateTime, nullable=True, comment="最后检测时间")
    remark: Mapped[Optional[str]] = mapped_column(Text, nullable=True, comment="备注")
    created_at: Mapped[datetime] = mapped_column(DateTime, default=datetime.now, comment="创建时间")
    updated_at: Mapped[datetime] = mapped_column(
        DateTime, 
        default=datetime.now, 
        onupdate=datetime.now,
        comment="更新时间"
    )
    
    # 关联
    customer: Mapped[Optional["Customer"]] = relationship(back_populates="socks5_proxies")
    nodes: Mapped[List["Node"]] = relationship(back_populates="socks5", cascade="all, delete-orphan")
    
    def __repr__(self):
        return f"<Socks5Proxy(id={self.id}, ip={self.ip}:{self.port})>"




