"""
订阅用户模型：一个节点可以分配给多个订阅用户
"""
from datetime import datetime
from enum import Enum
from sqlalchemy import Column, Integer, String, DateTime, BigInteger, Boolean, ForeignKey, Text
from sqlalchemy.orm import relationship
from ..core.database import Base


class SubscriberStatus(str, Enum):
    """订阅用户状态"""
    ACTIVE = "active"           # 正常使用
    DISABLED = "disabled"       # 已禁用
    EXPIRED = "expired"         # 已过期
    TRAFFIC_EXCEEDED = "traffic_exceeded"  # 流量超限


class Subscriber(Base):
    """订阅用户模型"""
    __tablename__ = "subscribers"
    
    id = Column(Integer, primary_key=True, index=True)
    node_id = Column(Integer, ForeignKey("nodes.id"), nullable=False, index=True)
    
    # 用户信息
    name = Column(String(100), nullable=False)  # 用户名称
    email = Column(String(255), nullable=True)  # 邮箱（可选，用于Xray流量统计标识）
    remark = Column(Text, nullable=True)        # 备注
    
    # Xray用户标识
    uuid = Column(String(36), nullable=False, unique=True, index=True)
    
    # 订阅链接
    subscribe_token = Column(String(64), nullable=False, unique=True, index=True)
    
    # 流量统计（字节）
    traffic_up = Column(BigInteger, default=0)      # 上行流量
    traffic_down = Column(BigInteger, default=0)    # 下行流量
    traffic_total = Column(BigInteger, default=0)   # 总流量
    traffic_limit = Column(BigInteger, nullable=True)  # 流量限制（None表示无限制）
    last_traffic_sync = Column(DateTime, nullable=True)  # 最后流量同步时间
    
    # 有效期
    expire_at = Column(DateTime, nullable=True)     # 到期时间（None表示永不过期）
    
    # 状态
    status = Column(String(20), default=SubscriberStatus.ACTIVE)
    is_enabled = Column(Boolean, default=True)      # 是否启用
    
    # 时间戳
    created_at = Column(DateTime, default=datetime.now)
    updated_at = Column(DateTime, default=datetime.now, onupdate=datetime.now)
    
    # 关联
    # node = relationship("Node", back_populates="subscribers")
    
    def __repr__(self):
        return f"<Subscriber {self.name} for Node {self.node_id}>"
    
    @property
    def is_expired(self) -> bool:
        """是否已过期"""
        if not self.expire_at:
            return False
        return datetime.now() >= self.expire_at
    
    @property
    def is_traffic_exceeded(self) -> bool:
        """是否流量超限"""
        if not self.traffic_limit:
            return False
        return self.traffic_total >= self.traffic_limit
    
    @property
    def xray_email(self) -> str:
        """用于Xray流量统计的email标识"""
        return f"sub_{self.id}@node_{self.node_id}"


