"""
客户Schema
"""
from datetime import datetime
from typing import Optional, List
from pydantic import BaseModel, Field

from ..models.customer import CustomerType


class CustomerBase(BaseModel):
    """客户基础字段"""
    name: str = Field(..., min_length=1, max_length=100, description="客户名称")
    contact: Optional[str] = Field(None, max_length=100, description="联系方式")
    customer_type: CustomerType = Field(CustomerType.LIVE, description="客户类型")
    expire_at: Optional[datetime] = Field(None, description="服务到期时间")
    remark: Optional[str] = Field(None, description="备注")


class CustomerCreate(CustomerBase):
    """创建客户"""
    pass


class CustomerUpdate(BaseModel):
    """更新客户"""
    name: Optional[str] = Field(None, min_length=1, max_length=100)
    contact: Optional[str] = Field(None, max_length=100)
    customer_type: Optional[CustomerType] = None
    expire_at: Optional[datetime] = None
    remark: Optional[str] = None


class CustomerResponse(CustomerBase):
    """客户响应"""
    id: int
    created_at: datetime
    updated_at: datetime
    node_count: int = 0
    # 客户级订阅
    subscribe_token: Optional[str] = None
    subscribe_url: Optional[str] = None
    
    class Config:
        from_attributes = True


class CustomerListResponse(BaseModel):
    """客户列表响应"""
    items: List[CustomerResponse]
    total: int




