"""
软件授权 Schema
"""
from datetime import datetime
from typing import Optional, List
from pydantic import BaseModel, Field

from ..models.license import LicenseType, LicenseStatus


# =============== License 卡密 ===============

class LicenseCreate(BaseModel):
    """创建卡密"""
    license_type: LicenseType = LicenseType.MONTHLY
    count: int = Field(default=1, ge=1, le=100, description="生成数量")
    max_devices: int = Field(default=1, ge=1, le=10, description="最大设备数")
    remark: Optional[str] = None


class LicenseUpdate(BaseModel):
    """更新卡密"""
    status: Optional[LicenseStatus] = None
    max_devices: Optional[int] = None
    remark: Optional[str] = None
    user_remark: Optional[str] = None


class LicenseResponse(BaseModel):
    """卡密响应"""
    id: int
    code: str
    license_type: LicenseType
    duration_days: int
    max_devices: int
    status: LicenseStatus
    activated_at: Optional[datetime] = None
    expire_at: Optional[datetime] = None
    device_fingerprint: Optional[str] = None
    device_info: Optional[str] = None
    user_remark: Optional[str] = None
    remark: Optional[str] = None
    days_remaining: int = 0
    created_at: datetime
    
    class Config:
        from_attributes = True


class LicenseListResponse(BaseModel):
    """卡密列表响应"""
    total: int
    items: List[LicenseResponse]


class LicenseGenerateResponse(BaseModel):
    """批量生成卡密响应"""
    count: int
    codes: List[str]


# =============== 激活相关 ===============

class ActivateRequest(BaseModel):
    """激活请求"""
    license_code: str = Field(..., description="授权码")
    device_fingerprint: str = Field(..., description="设备指纹")
    device_info: Optional[str] = Field(None, description="设备信息JSON")


class ActivateResponse(BaseModel):
    """激活响应"""
    success: bool
    message: str
    license_type: Optional[LicenseType] = None
    expire_at: Optional[datetime] = None
    days_remaining: int = 0


class VerifyRequest(BaseModel):
    """验证请求"""
    license_code: str
    device_fingerprint: str


class VerifyResponse(BaseModel):
    """验证响应"""
    valid: bool
    message: str
    license_type: Optional[LicenseType] = None
    expire_at: Optional[datetime] = None
    days_remaining: int = 0
    # 系统信息
    latest_version: str = "1.0.0"
    current_version_ok: bool = True
    notices: List[dict] = []


# =============== 系统公告 ===============

class NoticeCreate(BaseModel):
    """创建公告"""
    title: str
    content: str
    notice_type: str = "info"  # info/warning/danger
    is_force: bool = False
    start_at: Optional[datetime] = None
    end_at: Optional[datetime] = None


class NoticeUpdate(BaseModel):
    """更新公告"""
    title: Optional[str] = None
    content: Optional[str] = None
    notice_type: Optional[str] = None
    is_force: Optional[bool] = None
    is_active: Optional[bool] = None
    start_at: Optional[datetime] = None
    end_at: Optional[datetime] = None


class NoticeResponse(BaseModel):
    """公告响应"""
    id: int
    title: str
    content: str
    notice_type: str
    is_force: bool
    is_active: bool
    start_at: Optional[datetime] = None
    end_at: Optional[datetime] = None
    created_at: datetime
    
    class Config:
        from_attributes = True


class NoticeListResponse(BaseModel):
    """公告列表响应"""
    total: int
    items: List[NoticeResponse]


# =============== 系统配置 ===============

class SystemInfo(BaseModel):
    """系统信息响应（公开接口）"""
    app_name: str
    latest_version: str
    min_version: str
    download_url: str
    notices: List[NoticeResponse] = []

