"""
节点Schema
"""
from datetime import datetime
from typing import Optional, List
from pydantic import BaseModel, Field

from ..models.node import NodeProtocol, NodeStatus


class NodeDeployRequest(BaseModel):
    """部署节点请求"""
    customer_id: int = Field(..., description="客户ID")
    server_id: int = Field(..., description="中转服务器ID")
    socks5_id: int = Field(..., description="SK5落地ID")
    domain_id: Optional[int] = Field(None, description="域名ID（可选）")
    
    name: Optional[str] = Field(None, description="节点名称（不填自动生成）")
    protocol: NodeProtocol = Field(NodeProtocol.VLESS_REALITY, description="协议类型")
    listen_port: int = Field(443, ge=1, le=65535, description="监听端口")
    
    # Reality配置
    reality_dest: str = Field("www.apple.com:443", description="Reality目标")
    reality_server_names: List[str] = Field(["www.apple.com"], description="Reality SNI列表")


class NodeResponse(BaseModel):
    """节点响应"""
    id: int
    customer_id: int
    customer_name: str
    server_id: int
    server_name: str
    server_ip: str
    socks5_id: int
    socks5_info: str  # IP:端口
    domain_id: Optional[int]
    domain_name: Optional[str]
    
    name: str
    protocol: NodeProtocol
    listen_port: int
    uuid: str
    
    reality_dest: Optional[str]
    reality_public_key: Optional[str]
    reality_short_id: Optional[str]
    
    share_link: Optional[str]
    subscribe_token: Optional[str]
    
    status: NodeStatus
    latency: Optional[int]
    last_check: Optional[datetime]
    
    # 流量统计
    traffic_up: int = 0
    traffic_down: int = 0
    traffic_total: int = 0
    traffic_limit: Optional[int] = None
    last_traffic_sync: Optional[datetime] = None
    
    # 到期时间
    expire_at: Optional[datetime] = None
    is_expired: bool = False
    is_traffic_exceeded: bool = False
    
    created_at: datetime
    
    class Config:
        from_attributes = True


class NodeListResponse(BaseModel):
    """节点列表响应"""
    items: List[NodeResponse]
    total: int


class NodeShareInfo(BaseModel):
    """节点分享信息"""
    share_link: str
    qrcode_base64: str
    config_json: str


class SubscribeResponse(BaseModel):
    """订阅响应"""
    nodes: List[str]  # VLESS链接列表
    base64: str       # Base64编码的订阅内容


class NodeSettingsRequest(BaseModel):
    """节点设置请求"""
    traffic_limit: Optional[int] = Field(None, description="流量限制(字节)，0表示无限制")
    expire_at: Optional[datetime] = Field(None, description="到期时间")


class NodeTrafficInfo(BaseModel):
    """节点流量信息"""
    traffic_up: int = 0
    traffic_down: int = 0
    traffic_total: int = 0
    traffic_limit: Optional[int] = None
    traffic_used_percent: float = 0.0
    expire_at: Optional[datetime] = None
    days_remaining: Optional[int] = None
    is_expired: bool = False
    is_traffic_exceeded: bool = False
    last_sync: Optional[datetime] = None


class NodeResetTrafficRequest(BaseModel):
    """重置流量请求"""
    reset_traffic: bool = True



