"""
服务器Schema
"""
from datetime import datetime
from typing import Optional, List
from pydantic import BaseModel, Field

from ..models.server import ServerStatus


class ServerBase(BaseModel):
    """服务器基础字段"""
    name: str = Field(..., min_length=1, max_length=100, description="服务器名称")
    ip: str = Field(..., description="IP地址")
    ssh_port: int = Field(22, ge=1, le=65535, description="SSH端口")
    ssh_user: str = Field("root", description="SSH用户名")
    ssh_password: str = Field(..., description="SSH密码")
    region: Optional[str] = Field(None, description="地区")
    is_own: bool = Field(False, description="是否自有服务器")


class ServerCreate(ServerBase):
    """创建服务器"""
    customer_id: Optional[int] = Field(None, description="关联客户ID")


class ServerUpdate(BaseModel):
    """更新服务器"""
    name: Optional[str] = Field(None, min_length=1, max_length=100)
    ip: Optional[str] = None
    ssh_port: Optional[int] = Field(None, ge=1, le=65535)
    ssh_user: Optional[str] = None
    ssh_password: Optional[str] = None
    region: Optional[str] = None
    is_own: Optional[bool] = None


class ServerResponse(BaseModel):
    """服务器响应"""
    id: int
    customer_id: Optional[int]
    customer_name: Optional[str] = None
    name: str
    ip: str
    ssh_port: int
    ssh_user: str
    region: Optional[str]
    status: ServerStatus
    is_own: bool
    last_check: Optional[datetime]
    created_at: datetime
    node_count: int = 0
    
    class Config:
        from_attributes = True


class ServerListResponse(BaseModel):
    """服务器列表响应"""
    items: List[ServerResponse]
    total: int


class ServerParseRequest(BaseModel):
    """解析服务器信息请求"""
    text: str = Field(..., description="服务器信息文本")


class ServerParseResponse(BaseModel):
    """解析服务器信息响应"""
    ip: str
    ssh_port: int
    ssh_user: str
    ssh_password: str




