"""
SOCKS5 Schema
"""
from datetime import datetime
from typing import Optional, List
from pydantic import BaseModel, Field

from ..models.socks5 import Socks5Status


class Socks5Base(BaseModel):
    """SK5基础字段"""
    ip: str = Field(..., description="IP地址")
    port: int = Field(..., ge=1, le=65535, description="端口")
    username: Optional[str] = Field(None, description="用户名")
    password: Optional[str] = Field(None, description="密码")
    remark: Optional[str] = Field(None, description="备注")


class Socks5Create(Socks5Base):
    """创建SK5"""
    customer_id: Optional[int] = Field(None, description="关联客户ID")


class Socks5Update(BaseModel):
    """更新SK5"""
    ip: Optional[str] = None
    port: Optional[int] = Field(None, ge=1, le=65535)
    username: Optional[str] = None
    password: Optional[str] = None
    remark: Optional[str] = None


class Socks5Response(BaseModel):
    """SK5响应"""
    id: int
    customer_id: Optional[int]
    customer_name: Optional[str] = None
    ip: str
    port: int
    username: Optional[str]
    region: Optional[str]
    status: Socks5Status
    latency: Optional[int]
    last_check: Optional[datetime]
    remark: Optional[str]
    created_at: datetime
    node_count: int = 0
    
    class Config:
        from_attributes = True


class Socks5ListResponse(BaseModel):
    """SK5列表响应"""
    items: List[Socks5Response]
    total: int


class Socks5ParseRequest(BaseModel):
    """解析SK5信息请求"""
    text: str = Field(..., description="SK5信息文本（支持批量，每行一个）")


class Socks5ParseItem(BaseModel):
    """解析出的单个SK5"""
    ip: str
    port: int
    username: Optional[str] = None
    password: Optional[str] = None
    raw: str  # 原始文本


class Socks5ParseResponse(BaseModel):
    """解析SK5信息响应"""
    items: List[Socks5ParseItem]
    success_count: int
    fail_count: int


class Socks5BatchCreate(BaseModel):
    """批量创建SK5"""
    customer_id: Optional[int] = None
    items: List[Socks5Base]




