"""
订阅用户Schema
"""
from datetime import datetime
from typing import Optional, List
from pydantic import BaseModel, Field


class SubscriberBase(BaseModel):
    """订阅用户基础信息"""
    name: str = Field(..., min_length=1, max_length=100, description="用户名称")
    email: Optional[str] = Field(None, max_length=255, description="邮箱")
    remark: Optional[str] = Field(None, description="备注")
    traffic_limit: Optional[int] = Field(None, ge=0, description="流量限制（字节），None表示无限制")
    expire_at: Optional[datetime] = Field(None, description="到期时间，None表示永不过期")


class SubscriberCreate(SubscriberBase):
    """创建订阅用户"""
    node_id: int = Field(..., description="所属节点ID")


class SubscriberUpdate(BaseModel):
    """更新订阅用户"""
    name: Optional[str] = Field(None, min_length=1, max_length=100)
    email: Optional[str] = Field(None, max_length=255)
    remark: Optional[str] = None
    traffic_limit: Optional[int] = Field(None, ge=0)
    expire_at: Optional[datetime] = None
    is_enabled: Optional[bool] = None


class SubscriberResponse(SubscriberBase):
    """订阅用户响应"""
    id: int
    node_id: int
    uuid: str
    subscribe_token: str
    
    # 流量统计
    traffic_up: int = 0
    traffic_down: int = 0
    traffic_total: int = 0
    last_traffic_sync: Optional[datetime] = None
    
    # 状态
    status: str
    is_enabled: bool
    is_expired: bool = False
    is_traffic_exceeded: bool = False
    
    # 节点信息（用于显示）
    node_name: Optional[str] = None
    server_ip: Optional[str] = None
    listen_port: Optional[int] = None
    
    # 分享链接
    share_link: Optional[str] = None
    subscribe_url: Optional[str] = None
    
    created_at: datetime
    updated_at: datetime
    
    class Config:
        from_attributes = True


class SubscriberListResponse(BaseModel):
    """订阅用户列表响应"""
    items: List[SubscriberResponse]
    total: int


class SubscriberTrafficInfo(BaseModel):
    """订阅用户流量信息"""
    success: bool
    message: str = ""
    traffic_up: int = 0
    traffic_down: int = 0
    traffic_total: int = 0


class SubscriberSettingsRequest(BaseModel):
    """订阅用户设置请求"""
    traffic_limit: Optional[int] = Field(None, ge=0, description="流量限制（字节）")
    expire_at: Optional[datetime] = Field(None, description="到期时间")


class SubscriberBatchCreate(BaseModel):
    """批量创建订阅用户"""
    node_id: int
    count: int = Field(..., ge=1, le=100, description="创建数量")
    name_prefix: str = Field("用户", description="名称前缀")
    traffic_limit: Optional[int] = Field(None, ge=0, description="流量限制")
    expire_at: Optional[datetime] = Field(None, description="到期时间")


class SubscriberShareInfo(BaseModel):
    """订阅用户分享信息"""
    name: str
    share_link: str
    subscribe_url: str
    qrcode_base64: str


