"""
部署服务：部署节点到远程服务器
"""
import json
from typing import Optional, List, Tuple
from sqlalchemy.ext.asyncio import AsyncSession

from ..core.security import (
    decrypt_string, 
    generate_uuid, 
    generate_reality_keys,
    generate_short_id,
    generate_subscribe_token
)
from ..models import Node, Server, Socks5Proxy, Customer, Domain
from ..models.node import NodeProtocol, NodeStatus
from .ssh_service import SSHService
from .link_service import LinkService


class DeployService:
    """部署服务"""
    
    # 推荐的端口范围
    PORT_RANGE_START = 10000
    PORT_RANGE_END = 60000
    PREFERRED_PORTS = [443, 8443, 2053, 2083, 2087, 2096]  # 常用端口优先
    
    def __init__(self, db: AsyncSession):
        self.db = db
        self.ssh_service = SSHService()
        self.link_service = LinkService()
    
    async def get_used_ports(self, server: Server) -> List[int]:
        """获取服务器已占用的端口"""
        server_password = decrypt_string(server.ssh_password)
        
        # 获取所有监听的TCP端口
        command = "ss -tlnp | grep LISTEN | awk '{print $4}' | rev | cut -d: -f1 | rev | sort -n | uniq"
        
        success, stdout, stderr = await self.ssh_service.execute_command(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=server_password,
            command=command,
            timeout=30
        )
        
        if not success:
            # 备用命令
            command = "netstat -tlnp 2>/dev/null | grep LISTEN | awk '{print $4}' | rev | cut -d: -f1 | rev | sort -n | uniq"
            success, stdout, stderr = await self.ssh_service.execute_command(
                host=server.ip,
                port=server.ssh_port,
                username=server.ssh_user,
                password=server_password,
                command=command,
                timeout=30
            )
        
        used_ports = []
        if success and stdout:
            for line in stdout.strip().split('\n'):
                line = line.strip()
                if line.isdigit():
                    used_ports.append(int(line))
        
        return sorted(set(used_ports))
    
    async def find_available_port(self, server: Server, preferred_port: int = 443) -> int:
        """找到一个可用的端口"""
        used_ports = await self.get_used_ports(server)
        used_ports_set = set(used_ports)
        
        # 首先检查用户指定的端口
        if preferred_port not in used_ports_set:
            return preferred_port
        
        # 然后检查常用端口
        for port in self.PREFERRED_PORTS:
            if port not in used_ports_set:
                return port
        
        # 最后在范围内找一个可用端口
        for port in range(self.PORT_RANGE_START, self.PORT_RANGE_END):
            if port not in used_ports_set:
                return port
        
        raise Exception("无法找到可用端口")
    
    async def open_firewall_port(self, server: Server, port: int) -> Tuple[bool, str]:
        """自动放行防火墙端口"""
        server_password = decrypt_string(server.ssh_password)
        
        # 检测并放行端口的脚本
        firewall_script = f'''#!/bin/bash
PORT={port}

# 检测防火墙类型并放行端口
open_port() {{
    # 检查 ufw
    if command -v ufw &> /dev/null; then
        if ufw status | grep -q "Status: active"; then
            echo "检测到 UFW 防火墙，放行端口 $PORT..."
            ufw allow $PORT/tcp
            ufw reload
            echo "UFW: 端口 $PORT 已放行"
            return 0
        fi
    fi
    
    # 检查 firewalld
    if command -v firewall-cmd &> /dev/null; then
        if systemctl is-active firewalld &> /dev/null; then
            echo "检测到 Firewalld 防火墙，放行端口 $PORT..."
            firewall-cmd --permanent --add-port=$PORT/tcp
            firewall-cmd --reload
            echo "Firewalld: 端口 $PORT 已放行"
            return 0
        fi
    fi
    
    # 检查 iptables
    if command -v iptables &> /dev/null; then
        # 检查是否有 iptables 规则
        if iptables -L INPUT -n | grep -q "DROP\|REJECT"; then
            echo "检测到 iptables 防火墙，放行端口 $PORT..."
            iptables -I INPUT -p tcp --dport $PORT -j ACCEPT
            # 尝试保存规则
            if command -v iptables-save &> /dev/null; then
                iptables-save > /etc/iptables.rules 2>/dev/null || true
            fi
            if command -v netfilter-persistent &> /dev/null; then
                netfilter-persistent save 2>/dev/null || true
            fi
            echo "iptables: 端口 $PORT 已放行"
            return 0
        fi
    fi
    
    echo "未检测到活动的防火墙，或防火墙已允许所有流量"
    return 0
}}

open_port
'''
        
        success, stdout, stderr = await self.ssh_service.execute_script(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=server_password,
            script_content=firewall_script,
            timeout=60
        )
        
        message = stdout if success else stderr
        return success, message
    
    async def deploy(
        self,
        customer: Customer,
        server: Server,
        socks5: Socks5Proxy,
        domain: Optional[Domain],
        protocol: NodeProtocol,
        listen_port: int,
        name: Optional[str] = None,
        reality_dest: str = "www.apple.com:443",
        reality_server_names: List[str] = None,
        auto_port: bool = True  # 新增：是否自动选择端口
    ) -> Node:
        """部署节点"""
        
        if reality_server_names is None:
            reality_server_names = ["www.apple.com"]
        
        # 解密密码
        server_password = decrypt_string(server.ssh_password)
        socks5_password = decrypt_string(socks5.password) if socks5.password else None
        
        # 自动选择可用端口
        if auto_port:
            actual_port = await self.find_available_port(server, listen_port)
            if actual_port != listen_port:
                print(f"端口 {listen_port} 已被占用，自动切换到端口 {actual_port}")
            listen_port = actual_port
        else:
            # 检查端口是否可用
            used_ports = await self.get_used_ports(server)
            if listen_port in used_ports:
                raise Exception(f"端口 {listen_port} 已被占用，请选择其他端口")
        
        # 自动放行防火墙端口
        firewall_success, firewall_msg = await self.open_firewall_port(server, listen_port)
        if not firewall_success:
            print(f"防火墙放行警告: {firewall_msg}")
        
        # 生成配置
        uuid = generate_uuid()
        private_key, public_key = generate_reality_keys()
        short_id = generate_short_id()
        subscribe_token = generate_subscribe_token()
        
        # 节点名称
        if not name:
            name = f"{customer.name}-{server.region or server.ip}"
        
        # 生成Xray配置
        xray_config = self._generate_xray_config(
            uuid=uuid,
            listen_port=listen_port,
            protocol=protocol,
            socks5_ip=socks5.ip,
            socks5_port=socks5.port,
            socks5_user=socks5.username,
            socks5_pass=socks5_password,
            private_key=private_key,
            short_id=short_id,
            reality_dest=reality_dest,
            reality_server_names=reality_server_names
        )
        
        # 生成分享链接
        share_link = self.link_service.generate_vless_link(
            uuid=uuid,
            server_ip=server.ip,
            port=listen_port,
            node_name=name,
            protocol=protocol,
            reality_public_key=public_key,
            reality_short_id=short_id,
            reality_server_name=reality_server_names[0] if reality_server_names else "www.apple.com"
        )
        
        # 创建节点记录
        node = Node(
            customer_id=customer.id,
            server_id=server.id,
            socks5_id=socks5.id,
            domain_id=domain.id if domain else None,
            name=name,
            protocol=protocol,
            listen_port=listen_port,
            uuid=uuid,
            reality_dest=reality_dest,
            reality_server_names=json.dumps(reality_server_names),
            reality_private_key=private_key,
            reality_public_key=public_key,
            reality_short_id=short_id,
            config_json=json.dumps(xray_config, indent=2),
            share_link=share_link,
            subscribe_token=subscribe_token,
            status=NodeStatus.DEPLOYING
        )
        
        self.db.add(node)
        await self.db.flush()
        
        # 部署到远程服务器
        try:
            await self._deploy_to_server(
                server=server,
                server_password=server_password,
                xray_config=xray_config,
                node_id=node.id,
                listen_port=listen_port
            )
            node.status = NodeStatus.RUNNING
        except Exception as e:
            node.status = NodeStatus.ERROR
            raise e
        finally:
            await self.db.commit()
            await self.db.refresh(node)
        
        return node
    
    def _generate_xray_config(
        self,
        uuid: str,
        listen_port: int,
        protocol: NodeProtocol,
        socks5_ip: str,
        socks5_port: int,
        socks5_user: Optional[str],
        socks5_pass: Optional[str],
        private_key: str,
        short_id: str,
        reality_dest: str,
        reality_server_names: List[str]
    ) -> dict:
        """生成Xray配置"""
        
        # 出站配置（SOCKS5落地）
        outbound = {
            "protocol": "socks",
            "settings": {
                "servers": [
                    {
                        "address": socks5_ip,
                        "port": socks5_port
                    }
                ]
            },
            "tag": "proxy"
        }
        
        # 如果有认证
        if socks5_user and socks5_pass:
            outbound["settings"]["servers"][0]["users"] = [
                {
                    "user": socks5_user,
                    "pass": socks5_pass
                }
            ]
        
        # 入站配置
        if protocol == NodeProtocol.VLESS_REALITY:
            inbound = {
                "listen": "0.0.0.0",
                "port": listen_port,
                "protocol": "vless",
                "settings": {
                    "clients": [
                        {
                            "id": uuid,
                            "flow": "xtls-rprx-vision"
                        }
                    ],
                    "decryption": "none"
                },
                "streamSettings": {
                    "network": "tcp",
                    "security": "reality",
                    "realitySettings": {
                        "show": False,
                        "dest": reality_dest,
                        "xver": 0,
                        "serverNames": reality_server_names,
                        "privateKey": private_key,
                        "shortIds": [short_id]
                    }
                },
                "tag": "vless-in"
            }
        else:
            # VLESS + TLS + WS
            inbound = {
                "listen": "0.0.0.0",
                "port": listen_port,
                "protocol": "vless",
                "settings": {
                    "clients": [
                        {
                            "id": uuid
                        }
                    ],
                    "decryption": "none"
                },
                "streamSettings": {
                    "network": "ws",
                    "security": "tls",
                    "wsSettings": {
                        "path": "/ws"
                    }
                },
                "tag": "vless-in"
            }
        
        config = {
            "log": {
                "loglevel": "warning"
            },
            # 流量统计配置
            "stats": {},
            "api": {
                "tag": "api",
                "services": ["StatsService"]
            },
            "policy": {
                "levels": {
                    "0": {
                        "statsUserUplink": True,
                        "statsUserDownlink": True
                    }
                },
                "system": {
                    "statsInboundUplink": True,
                    "statsInboundDownlink": True,
                    "statsOutboundUplink": True,
                    "statsOutboundDownlink": True
                }
            },
            "inbounds": [
                # API入站（本地访问流量统计）
                {
                    "listen": "127.0.0.1",
                    "port": 10085,
                    "protocol": "dokodemo-door",
                    "settings": {
                        "address": "127.0.0.1"
                    },
                    "tag": "api"
                },
                inbound
            ],
            "outbounds": [
                outbound,
                {
                    "protocol": "freedom",
                    "tag": "direct"
                },
                {
                    "protocol": "blackhole",
                    "tag": "block"
                }
            ],
            "routing": {
                "rules": [
                    {
                        "type": "field",
                        "inboundTag": ["api"],
                        "outboundTag": "api"
                    },
                    {
                        "type": "field",
                        "inboundTag": ["vless-in"],
                        "outboundTag": "proxy"
                    }
                ]
            }
        }
        
        return config
    
    async def _deploy_to_server(
        self,
        server: Server,
        server_password: str,
        xray_config: dict,
        node_id: int,
        listen_port: int
    ):
        """部署到远程服务器"""
        
        config_json = json.dumps(xray_config, indent=2)
        config_path = f"/etc/xray/config_{node_id}.json"
        service_name = f"xray-node-{node_id}"
        
        # 安装脚本（包含防火墙放行）
        install_script = f'''#!/bin/bash
set -e

PORT={listen_port}

echo "=========================================="
echo "开始部署节点 #{node_id}"
echo "监听端口: $PORT"
echo "=========================================="

# 安装Xray（如果未安装）
if ! command -v xray &> /dev/null; then
    echo "[1/5] 安装 Xray..."
    bash -c "$(curl -L https://github.com/XTLS/Xray-install/raw/main/install-release.sh)" @ install
else
    echo "[1/5] Xray 已安装，跳过"
fi

# 创建配置目录
echo "[2/5] 创建配置文件..."
mkdir -p /etc/xray

# 写入配置文件
cat > {config_path} << 'XRAY_CONFIG'
{config_json}
XRAY_CONFIG

# 创建systemd服务
echo "[3/5] 创建系统服务..."
cat > /etc/systemd/system/{service_name}.service << 'SYSTEMD_SERVICE'
[Unit]
Description=Xray Node {node_id}
After=network.target

[Service]
Type=simple
ExecStart=/usr/local/bin/xray run -config {config_path}
Restart=on-failure
RestartSec=5

[Install]
WantedBy=multi-user.target
SYSTEMD_SERVICE

# 放行防火墙端口
echo "[4/5] 配置防火墙..."
if command -v ufw &> /dev/null && ufw status | grep -q "Status: active"; then
    ufw allow $PORT/tcp >/dev/null 2>&1 && echo "    UFW: 已放行端口 $PORT"
elif command -v firewall-cmd &> /dev/null && systemctl is-active firewalld &> /dev/null; then
    firewall-cmd --permanent --add-port=$PORT/tcp >/dev/null 2>&1
    firewall-cmd --reload >/dev/null 2>&1 && echo "    Firewalld: 已放行端口 $PORT"
elif command -v iptables &> /dev/null; then
    iptables -I INPUT -p tcp --dport $PORT -j ACCEPT 2>/dev/null && echo "    iptables: 已放行端口 $PORT"
else
    echo "    未检测到防火墙或已全部放行"
fi

# 重载并启动服务
echo "[5/5] 启动服务..."
systemctl daemon-reload
systemctl enable {service_name} >/dev/null 2>&1
systemctl restart {service_name}

# 检查状态
sleep 2
if systemctl is-active {service_name} >/dev/null 2>&1; then
    echo ""
    echo "=========================================="
    echo "✅ 部署成功！"
    echo "   节点ID: {node_id}"
    echo "   端口: $PORT"
    echo "   服务: {service_name}"
    echo "=========================================="
else
    echo "❌ 服务启动失败"
    systemctl status {service_name}
    exit 1
fi
'''
        
        # 执行部署脚本
        success, stdout, stderr = await self.ssh_service.execute_script(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=server_password,
            script_content=install_script,
            timeout=300
        )
        
        if not success:
            raise Exception(f"部署脚本执行失败: {stderr}")
    
    async def restart_node(self, server: Server, node: Node):
        """重启节点"""
        server_password = decrypt_string(server.ssh_password)
        service_name = f"xray-node-{node.id}"
        
        success, stdout, stderr = await self.ssh_service.execute_command(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=server_password,
            command=f"systemctl restart {service_name}"
        )
        
        if not success:
            raise Exception(f"重启失败: {stderr}")
    
    async def stop_node(self, server: Server, node: Node):
        """停止节点"""
        server_password = decrypt_string(server.ssh_password)
        service_name = f"xray-node-{node.id}"
        
        success, stdout, stderr = await self.ssh_service.execute_command(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=server_password,
            command=f"systemctl stop {service_name}"
        )
        
        if not success:
            raise Exception(f"停止失败: {stderr}")
    
    async def upgrade_node_config(self, node: Node, server: Server, socks5: Socks5Proxy) -> bool:
        """
        升级节点配置 - 为旧节点添加流量统计功能
        
        此方法会重新生成Xray配置文件（包含stats和api模块），
        更新服务器上的配置并重启服务。
        """
        server_password = decrypt_string(server.ssh_password)
        if not server_password:
            server_password = server.ssh_password
        
        socks5_password = decrypt_string(socks5.password)
        if not socks5_password:
            socks5_password = socks5.password
        
        # 生成新的配置（包含stats和api）
        xray_config = self._generate_xray_config(
            uuid=node.uuid,
            listen_port=node.listen_port,
            protocol=node.protocol,
            socks5_ip=socks5.ip,
            socks5_port=socks5.port,
            socks5_user=socks5.username,
            socks5_pass=socks5_password,
            private_key=node.reality_private_key,
            short_id=node.reality_short_id,
            reality_dest=node.reality_dest,
            reality_server_names=node.reality_server_names.split(",") if node.reality_server_names else ["www.apple.com"]
        )
        
        config_json = json.dumps(xray_config, indent=2)
        config_path = f"/etc/xray/config_{node.id}.json"
        service_name = f"xray-node-{node.id}"
        
        # 更新配置脚本
        update_script = f'''#!/bin/bash
set -e

echo "=========================================="
echo "升级节点配置 #{node.id} - 启用流量统计"
echo "=========================================="

# 备份旧配置
if [ -f "{config_path}" ]; then
    cp {config_path} {config_path}.bak.$(date +%Y%m%d%H%M%S)
    echo "✓ 已备份旧配置"
fi

# 写入新配置
cat > {config_path} << 'XRAY_CONFIG'
{config_json}
XRAY_CONFIG

echo "✓ 已更新配置文件"

# 验证配置
if /usr/local/bin/xray run -test -config {config_path}; then
    echo "✓ 配置验证通过"
else
    echo "✗ 配置验证失败"
    exit 1
fi

# 重启服务
systemctl restart {service_name}
sleep 2

# 检查服务状态
if systemctl is-active --quiet {service_name}; then
    echo "✓ 服务已重启"
else
    echo "✗ 服务启动失败"
    systemctl status {service_name}
    exit 1
fi

# 检查API端口是否监听
sleep 1
if ss -tlnp | grep -q ":10085"; then
    echo "✓ API端口已监听 (10085)"
else
    echo "⚠ API端口未监听，流量统计可能不可用"
fi

echo "=========================================="
echo "✓ 节点配置升级完成！"
echo "=========================================="
'''
        
        success, stdout, stderr = await self.ssh_service.execute_script(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=server_password,
            script_content=update_script,
            timeout=60
        )
        
        if success:
            return True
        else:
            raise Exception(f"升级失败: {stderr}")
    
    async def restore_node_config(self, node: Node, server: Server) -> bool:
        """
        恢复节点配置 - 从备份恢复旧配置
        """
        server_password = decrypt_string(server.ssh_password)
        if not server_password:
            server_password = server.ssh_password
        
        config_path = f"/etc/xray/config_{node.id}.json"
        service_name = f"xray-node-{node.id}"
        
        # 恢复脚本
        restore_script = f'''#!/bin/bash
set -e

echo "=========================================="
echo "恢复节点配置 #{node.id}"
echo "=========================================="

# 查找最新的备份文件
BACKUP=$(ls -t {config_path}.bak.* 2>/dev/null | head -1)

if [ -z "$BACKUP" ]; then
    echo "✗ 没有找到备份文件"
    exit 1
fi

echo "找到备份: $BACKUP"

# 恢复备份
cp "$BACKUP" {config_path}
echo "✓ 已恢复配置"

# 重启服务
systemctl restart {service_name}
sleep 2

# 检查服务状态
if systemctl is-active --quiet {service_name}; then
    echo "✓ 服务已重启"
else
    echo "✗ 服务启动失败"
    exit 1
fi

echo "=========================================="
echo "✓ 节点配置恢复完成！"
echo "=========================================="
'''
        
        success, stdout, stderr = await self.ssh_service.execute_script(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=server_password,
            script_content=restore_script,
            timeout=60
        )
        
        if success:
            return True
        else:
            raise Exception(f"恢复失败: {stderr}")