"""
链接服务：生成分享链接和二维码
"""
import base64
import json
import urllib.parse
from typing import Optional, List
import io

import qrcode
from qrcode.image.pure import PyPNGImage

from ..models.node import Node, NodeProtocol


class LinkService:
    """链接服务"""
    
    def generate_vless_link(
        self,
        uuid: str,
        server_ip: str,
        port: int,
        node_name: str,
        protocol: NodeProtocol,
        reality_public_key: Optional[str] = None,
        reality_short_id: Optional[str] = None,
        reality_server_name: Optional[str] = None,
        reality_dest: Optional[str] = None
    ) -> str:
        """生成VLESS分享链接"""
        
        # 基础参数
        params = {
            "type": "tcp",
            "encryption": "none"
        }
        
        if protocol == NodeProtocol.VLESS_REALITY:
            # Reality协议参数
            params.update({
                "security": "reality",
                "pbk": reality_public_key or "",
                "sid": reality_short_id or "",
                "sni": reality_server_name or "www.apple.com",
                "fp": "chrome",
                "flow": "xtls-rprx-vision"
            })
        elif protocol == NodeProtocol.VLESS_TLS_WS:
            # TLS+WS协议参数
            params.update({
                "security": "tls",
                "type": "ws",
                "sni": reality_server_name or server_ip,
                "path": "/ws"
            })
        
        # 构建链接
        query = urllib.parse.urlencode(params)
        name = urllib.parse.quote(node_name)
        
        link = f"vless://{uuid}@{server_ip}:{port}?{query}#{name}"
        
        return link
    
    def generate_qrcode(self, content: str) -> str:
        """生成二维码Base64"""
        if not content:
            return ""
        
        try:
            qr = qrcode.QRCode(
                version=1,
                error_correction=qrcode.constants.ERROR_CORRECT_L,
                box_size=10,
                border=4,
            )
            qr.add_data(content)
            qr.make(fit=True)
            
            img = qr.make_image(fill_color="black", back_color="white")
            
            # 转换为Base64
            buffer = io.BytesIO()
            img.save(buffer, format='PNG')
            buffer.seek(0)
            
            img_base64 = base64.b64encode(buffer.getvalue()).decode()
            
            return f"data:image/png;base64,{img_base64}"
            
        except Exception as e:
            return ""
    
    def generate_clash_config(self, nodes: List[Node], name: str = "TikTok节点") -> str:
        """生成Clash配置"""
        proxies = []
        proxy_names = []
        
        for node in nodes:
            if not node.share_link:
                continue
            
            proxy = {
                "name": node.name,
                "type": "vless",
                "server": node.server.ip if node.server else "",
                "port": node.listen_port,
                "uuid": node.uuid,
                "network": "tcp",
                "tls": True,
            }
            
            if node.protocol == NodeProtocol.VLESS_REALITY:
                proxy.update({
                    "reality-opts": {
                        "public-key": node.reality_public_key,
                        "short-id": node.reality_short_id
                    },
                    "servername": node.reality_dest.split(":")[0] if node.reality_dest else "www.apple.com",
                    "flow": "xtls-rprx-vision",
                    "client-fingerprint": "chrome"
                })
            
            proxies.append(proxy)
            proxy_names.append(node.name)
        
        config = {
            "proxies": proxies,
            "proxy-groups": [
                {
                    "name": name,
                    "type": "select",
                    "proxies": proxy_names
                }
            ],
            "rules": [
                "MATCH," + name
            ]
        }
        
        import yaml
        return yaml.dump(config, allow_unicode=True, default_flow_style=False)
    
    def generate_subscribe_content(self, nodes: List[Node]) -> str:
        """生成订阅内容（Base64编码的链接列表）"""
        links = [node.share_link for node in nodes if node.share_link]
        content = "\n".join(links)
        return base64.b64encode(content.encode()).decode()




