"""
解析服务：解析服务器信息和SK5信息
"""
import re
from typing import Optional, Dict, List
from ..schemas.socks5 import Socks5ParseItem, Socks5ParseResponse


class ParserService:
    """解析服务"""
    
    @staticmethod
    def parse_server_info(text: str) -> Optional[Dict]:
        """
        解析服务器信息
        支持格式:
        1. IP: xxx SSH端口: xx 用户名: xxx 密码: xxx
        2. xxx.xxx.xxx.xxx:22:root:password
        3. xxx.xxx.xxx.xxx root password
        """
        text = text.strip()
        
        # 格式1: 标准格式（带标签）
        ip_match = re.search(r'IP[:\s：]+(\d+\.\d+\.\d+\.\d+)', text, re.IGNORECASE)
        port_match = re.search(r'(?:SSH)?[端口port]*[:\s：]+(\d+)', text, re.IGNORECASE)
        user_match = re.search(r'(?:用户名|user|username)[:\s：]+(\S+)', text, re.IGNORECASE)
        pwd_match = re.search(r'(?:密码|pass|password)[:\s：]+(\S+)', text, re.IGNORECASE)
        
        if ip_match:
            return {
                "ip": ip_match.group(1),
                "ssh_port": int(port_match.group(1)) if port_match else 22,
                "ssh_user": user_match.group(1) if user_match else "root",
                "ssh_password": pwd_match.group(1) if pwd_match else ""
            }
        
        # 格式2: IP:端口:用户名:密码
        match = re.match(r'^(\d+\.\d+\.\d+\.\d+):(\d+):(\S+):(\S+)$', text)
        if match:
            return {
                "ip": match.group(1),
                "ssh_port": int(match.group(2)),
                "ssh_user": match.group(3),
                "ssh_password": match.group(4)
            }
        
        # 格式3: IP 用户名 密码（端口默认22）
        match = re.match(r'^(\d+\.\d+\.\d+\.\d+)\s+(\S+)\s+(\S+)$', text)
        if match:
            return {
                "ip": match.group(1),
                "ssh_port": 22,
                "ssh_user": match.group(2),
                "ssh_password": match.group(3)
            }
        
        # 格式4: 只有IP
        match = re.match(r'^(\d+\.\d+\.\d+\.\d+)$', text)
        if match:
            return {
                "ip": match.group(1),
                "ssh_port": 22,
                "ssh_user": "root",
                "ssh_password": ""
            }
        
        return None
    
    @staticmethod
    def parse_socks5_info(text: str) -> Socks5ParseResponse:
        """
        解析SK5信息（支持批量）
        支持格式:
        1. IP:端口:用户名:密码
        2. IP:端口（无认证）
        3. socks5://用户名:密码@IP:端口
        """
        lines = text.strip().split('\n')
        items: List[Socks5ParseItem] = []
        fail_count = 0
        
        for line in lines:
            line = line.strip()
            if not line:
                continue
            
            result = ParserService._parse_single_socks5(line)
            if result:
                items.append(result)
            else:
                fail_count += 1
        
        return Socks5ParseResponse(
            items=items,
            success_count=len(items),
            fail_count=fail_count
        )
    
    @staticmethod
    def _parse_single_socks5(text: str) -> Optional[Socks5ParseItem]:
        """解析单个SK5"""
        text = text.strip()
        
        # 格式1: socks5://用户名:密码@IP:端口
        match = re.match(r'^socks5?://(?:(\S+):(\S+)@)?(\d+\.\d+\.\d+\.\d+):(\d+)$', text, re.IGNORECASE)
        if match:
            return Socks5ParseItem(
                ip=match.group(3),
                port=int(match.group(4)),
                username=match.group(1),
                password=match.group(2),
                raw=text
            )
        
        # 格式2: IP:端口:用户名:密码
        match = re.match(r'^(\d+\.\d+\.\d+\.\d+):(\d+):(\S+):(\S+)$', text)
        if match:
            return Socks5ParseItem(
                ip=match.group(1),
                port=int(match.group(2)),
                username=match.group(3),
                password=match.group(4),
                raw=text
            )
        
        # 格式3: IP:端口（无认证）
        match = re.match(r'^(\d+\.\d+\.\d+\.\d+):(\d+)$', text)
        if match:
            return Socks5ParseItem(
                ip=match.group(1),
                port=int(match.group(2)),
                username=None,
                password=None,
                raw=text
            )
        
        return None




