"""
服务器初始化服务：配置新服务器
"""
from typing import Optional, Dict, List
from dataclasses import dataclass
from enum import Enum

from .ssh_service import SSHService


class InitStep(str, Enum):
    """初始化步骤"""
    UPDATE_SYSTEM = "update_system"
    INSTALL_TOOLS = "install_tools"
    SET_TIMEZONE = "set_timezone"
    ENABLE_BBR = "enable_bbr"
    OPTIMIZE_KERNEL = "optimize_kernel"
    SETUP_FIREWALL = "setup_firewall"
    SECURE_SSH = "secure_ssh"
    INSTALL_XRAY = "install_xray"


@dataclass
class InitResult:
    """初始化结果"""
    success: bool
    step: str
    message: str
    output: str = ""


class ServerInitService:
    """服务器初始化服务"""
    
    def __init__(self):
        self.ssh_service = SSHService()
    
    async def init_server(
        self,
        host: str,
        port: int,
        username: str,
        password: str,
        steps: List[InitStep] = None,
        new_ssh_port: int = None,
        timezone: str = "Asia/Shanghai"
    ) -> List[InitResult]:
        """
        初始化服务器
        
        Args:
            host: 服务器IP
            port: SSH端口
            username: 用户名
            password: 密码
            steps: 要执行的步骤，None表示全部
            new_ssh_port: 新的SSH端口（可选）
            timezone: 时区
        """
        results = []
        
        if steps is None:
            steps = [
                InitStep.UPDATE_SYSTEM,
                InitStep.INSTALL_TOOLS,
                InitStep.SET_TIMEZONE,
                InitStep.ENABLE_BBR,
                InitStep.OPTIMIZE_KERNEL,
                InitStep.INSTALL_XRAY
            ]
        
        for step in steps:
            result = await self._execute_step(
                host=host,
                port=port,
                username=username,
                password=password,
                step=step,
                new_ssh_port=new_ssh_port,
                timezone=timezone
            )
            results.append(result)
            
            # 如果某步失败，继续执行其他步骤但标记
            if not result.success:
                print(f"步骤 {step} 失败: {result.message}")
        
        return results
    
    async def _execute_step(
        self,
        host: str,
        port: int,
        username: str,
        password: str,
        step: InitStep,
        new_ssh_port: int = None,
        timezone: str = "Asia/Shanghai"
    ) -> InitResult:
        """执行单个步骤"""
        
        step_scripts = {
            InitStep.UPDATE_SYSTEM: self._get_update_script(),
            InitStep.INSTALL_TOOLS: self._get_tools_script(),
            InitStep.SET_TIMEZONE: self._get_timezone_script(timezone),
            InitStep.ENABLE_BBR: self._get_bbr_script(),
            InitStep.OPTIMIZE_KERNEL: self._get_optimize_script(),
            InitStep.SETUP_FIREWALL: self._get_firewall_script(),
            InitStep.SECURE_SSH: self._get_ssh_secure_script(new_ssh_port),
            InitStep.INSTALL_XRAY: self._get_xray_script()
        }
        
        script = step_scripts.get(step)
        if not script:
            return InitResult(
                success=False,
                step=step.value,
                message="未知的步骤"
            )
        
        success, stdout, stderr = await self.ssh_service.execute_script(
            host=host,
            port=port,
            username=username,
            password=password,
            script_content=script,
            timeout=600  # 有些步骤需要较长时间
        )
        
        step_names = {
            InitStep.UPDATE_SYSTEM: "更新系统",
            InitStep.INSTALL_TOOLS: "安装常用工具",
            InitStep.SET_TIMEZONE: "设置时区",
            InitStep.ENABLE_BBR: "启用BBR加速",
            InitStep.OPTIMIZE_KERNEL: "优化内核参数",
            InitStep.SETUP_FIREWALL: "配置防火墙",
            InitStep.SECURE_SSH: "SSH安全配置",
            InitStep.INSTALL_XRAY: "安装Xray"
        }
        
        return InitResult(
            success=success,
            step=step_names.get(step, step.value),
            message="完成" if success else f"失败: {stderr[:200]}",
            output=stdout if success else stderr
        )
    
    def _get_update_script(self) -> str:
        """更新系统脚本"""
        return '''#!/bin/bash
echo "========== 更新系统 =========="

# 检测系统类型
if [ -f /etc/debian_version ]; then
    echo "检测到 Debian/Ubuntu 系统"
    export DEBIAN_FRONTEND=noninteractive
    apt-get update -y
    apt-get upgrade -y -o Dpkg::Options::="--force-confdef" -o Dpkg::Options::="--force-confold"
    apt-get autoremove -y
    apt-get autoclean -y
elif [ -f /etc/redhat-release ]; then
    echo "检测到 CentOS/RHEL 系统"
    yum update -y
    yum autoremove -y
else
    echo "未知系统类型"
    exit 1
fi

echo "✓ 系统更新完成"
'''
    
    def _get_tools_script(self) -> str:
        """安装常用工具脚本"""
        return '''#!/bin/bash
echo "========== 安装常用工具 =========="

if [ -f /etc/debian_version ]; then
    apt-get install -y curl wget vim net-tools htop unzip tar socat
elif [ -f /etc/redhat-release ]; then
    yum install -y curl wget vim net-tools htop unzip tar socat
fi

echo "✓ 常用工具安装完成"
'''
    
    def _get_timezone_script(self, timezone: str) -> str:
        """设置时区脚本"""
        return f'''#!/bin/bash
echo "========== 设置时区 =========="

timedatectl set-timezone {timezone}
echo "当前时间: $(date)"

# 同步时间
if command -v ntpdate &> /dev/null; then
    ntpdate pool.ntp.org || true
fi

echo "✓ 时区设置完成: {timezone}"
'''
    
    def _get_bbr_script(self) -> str:
        """启用BBR加速脚本"""
        return '''#!/bin/bash
echo "========== 启用BBR加速 =========="

# 检查内核版本
KERNEL_VERSION=$(uname -r | cut -d. -f1)
if [ "$KERNEL_VERSION" -lt 4 ]; then
    echo "内核版本过低，需要4.9+才能使用BBR"
    exit 0
fi

# 检查是否已启用
if lsmod | grep -q bbr; then
    echo "BBR 已经启用"
    exit 0
fi

# 启用BBR
cat >> /etc/sysctl.conf << EOF

# BBR
net.core.default_qdisc=fq
net.ipv4.tcp_congestion_control=bbr
EOF

sysctl -p

# 验证
if sysctl net.ipv4.tcp_congestion_control | grep -q bbr; then
    echo "✓ BBR加速启用成功"
else
    echo "BBR启用可能需要重启生效"
fi
'''
    
    def _get_optimize_script(self) -> str:
        """内核参数优化脚本"""
        return '''#!/bin/bash
echo "========== 优化内核参数 =========="

# 备份原配置
cp /etc/sysctl.conf /etc/sysctl.conf.bak.$(date +%Y%m%d)

# 添加优化参数
cat >> /etc/sysctl.conf << 'EOF'

# ===== 网络优化 =====
# 最大文件打开数
fs.file-max = 1048576

# TCP缓冲区
net.core.rmem_max = 16777216
net.core.wmem_max = 16777216
net.core.rmem_default = 262144
net.core.wmem_default = 262144
net.ipv4.tcp_rmem = 4096 87380 16777216
net.ipv4.tcp_wmem = 4096 65536 16777216

# 连接数
net.core.somaxconn = 65535
net.core.netdev_max_backlog = 65535
net.ipv4.tcp_max_syn_backlog = 65535

# TIME_WAIT优化
net.ipv4.tcp_tw_reuse = 1
net.ipv4.tcp_fin_timeout = 30
net.ipv4.tcp_max_tw_buckets = 5000

# 保活
net.ipv4.tcp_keepalive_time = 1200
net.ipv4.tcp_keepalive_intvl = 30
net.ipv4.tcp_keepalive_probes = 3

# 其他
net.ipv4.ip_local_port_range = 1024 65535
net.ipv4.tcp_fastopen = 3
EOF

sysctl -p

echo "✓ 内核参数优化完成"
'''
    
    def _get_firewall_script(self) -> str:
        """配置防火墙脚本"""
        return '''#!/bin/bash
echo "========== 配置防火墙 =========="

# 检测并配置防火墙
if command -v ufw &> /dev/null; then
    echo "配置 UFW..."
    ufw default deny incoming
    ufw default allow outgoing
    ufw allow ssh
    ufw allow 80/tcp
    ufw allow 443/tcp
    echo "y" | ufw enable
    echo "✓ UFW 配置完成"
    
elif command -v firewall-cmd &> /dev/null; then
    echo "配置 Firewalld..."
    systemctl start firewalld
    systemctl enable firewalld
    firewall-cmd --permanent --add-service=ssh
    firewall-cmd --permanent --add-port=80/tcp
    firewall-cmd --permanent --add-port=443/tcp
    firewall-cmd --reload
    echo "✓ Firewalld 配置完成"
    
else
    echo "未检测到防火墙，跳过配置"
fi
'''
    
    def _get_ssh_secure_script(self, new_port: int = None) -> str:
        """SSH安全配置脚本"""
        port_config = f"Port {new_port}" if new_port else ""
        firewall_cmd = ""
        
        if new_port:
            firewall_cmd = f'''
# 放行新SSH端口
if command -v ufw &> /dev/null; then
    ufw allow {new_port}/tcp
elif command -v firewall-cmd &> /dev/null; then
    firewall-cmd --permanent --add-port={new_port}/tcp
    firewall-cmd --reload
fi
'''
        
        return f'''#!/bin/bash
echo "========== SSH安全配置 =========="

# 备份配置
cp /etc/ssh/sshd_config /etc/ssh/sshd_config.bak.$(date +%Y%m%d)

# 修改配置
sed -i 's/#PermitRootLogin.*/PermitRootLogin yes/' /etc/ssh/sshd_config
sed -i 's/PermitRootLogin.*/PermitRootLogin yes/' /etc/ssh/sshd_config
sed -i 's/#MaxAuthTries.*/MaxAuthTries 3/' /etc/ssh/sshd_config
sed -i 's/#ClientAliveInterval.*/ClientAliveInterval 60/' /etc/ssh/sshd_config
sed -i 's/#ClientAliveCountMax.*/ClientAliveCountMax 3/' /etc/ssh/sshd_config

{f"# 修改SSH端口" if new_port else ""}
{f"sed -i 's/^#*Port.*/Port {new_port}/' /etc/ssh/sshd_config" if new_port else ""}

{firewall_cmd}

# 重启SSH
systemctl restart sshd

echo "✓ SSH安全配置完成"
{f'echo "⚠️ SSH端口已修改为: {new_port}"' if new_port else ''}
'''
    
    def _get_xray_script(self) -> str:
        """安装Xray脚本"""
        return '''#!/bin/bash
echo "========== 安装Xray =========="

if command -v xray &> /dev/null; then
    echo "Xray 已安装，检查更新..."
    bash -c "$(curl -L https://github.com/XTLS/Xray-install/raw/main/install-release.sh)" @ install
else
    echo "安装 Xray..."
    bash -c "$(curl -L https://github.com/XTLS/Xray-install/raw/main/install-release.sh)" @ install
fi

# 显示版本
xray version

echo "✓ Xray 安装完成"
'''
    
    async def get_server_info(
        self,
        host: str,
        port: int,
        username: str,
        password: str
    ) -> Dict:
        """获取服务器信息"""
        
        script = '''#!/bin/bash
echo "===INFO_START==="
echo "hostname:$(hostname)"
echo "os:$(cat /etc/os-release | grep PRETTY_NAME | cut -d'"' -f2)"
echo "kernel:$(uname -r)"
echo "cpu:$(grep -c processor /proc/cpuinfo) cores"
echo "memory:$(free -h | awk '/Mem:/{print $2}')"
echo "disk:$(df -h / | awk 'NR==2{print $2}')"
echo "uptime:$(uptime -p)"
echo "ip:$(curl -s ip.sb || curl -s ifconfig.me || echo 'unknown')"

# 检查BBR
if sysctl net.ipv4.tcp_congestion_control 2>/dev/null | grep -q bbr; then
    echo "bbr:enabled"
else
    echo "bbr:disabled"
fi

# 检查Xray
if command -v xray &> /dev/null; then
    echo "xray:$(xray version | head -1)"
else
    echo "xray:not installed"
fi

# 检查防火墙
if command -v ufw &> /dev/null && ufw status | grep -q "Status: active"; then
    echo "firewall:ufw (active)"
elif command -v firewall-cmd &> /dev/null && systemctl is-active firewalld &> /dev/null; then
    echo "firewall:firewalld (active)"
else
    echo "firewall:none or disabled"
fi

echo "===INFO_END==="
'''
        
        success, stdout, stderr = await self.ssh_service.execute_script(
            host=host,
            port=port,
            username=username,
            password=password,
            script_content=script,
            timeout=60
        )
        
        if not success:
            return {"error": stderr}
        
        # 解析输出
        info = {}
        in_info = False
        for line in stdout.split('\n'):
            if '===INFO_START===' in line:
                in_info = True
                continue
            if '===INFO_END===' in line:
                break
            if in_info and ':' in line:
                key, value = line.split(':', 1)
                info[key.strip()] = value.strip()
        
        return info




