"""
SOCKS5服务：测试SK5连接、IP检测
整合多个免费API：ip-api.com, ipinfo.io, ipapi.co
"""
import asyncio
import time
from typing import Optional, Dict, Any, List
from dataclasses import dataclass

import aiohttp
from aiohttp_socks import ProxyConnector

from ..schemas.common import TestResult


@dataclass
class IPCheckResult:
    """IP检测结果"""
    ip: str
    country: str = ""
    country_code: str = ""
    region: str = ""
    city: str = ""
    isp: str = ""
    org: str = ""
    as_name: str = ""
    # IP类型评估
    ip_type: str = "unknown"  # residential/datacenter/mobile/proxy
    is_proxy: bool = False
    is_vpn: bool = False
    is_datacenter: bool = False
    is_residential: bool = False
    # TikTok适用性评分 (0-100)
    tiktok_score: int = 0
    tiktok_risk: str = "unknown"  # low/medium/high
    risk_reasons: list = None
    
    def __post_init__(self):
        if self.risk_reasons is None:
            self.risk_reasons = []


class Socks5Service:
    """SOCKS5服务"""
    
    # 已知数据中心ASN关键词
    DATACENTER_KEYWORDS = [
        'amazon', 'aws', 'google', 'microsoft', 'azure', 'digitalocean',
        'linode', 'vultr', 'ovh', 'hetzner', 'cloudflare', 'akamai',
        'hosting', 'server', 'vps', 'cloud', 'datacenter', 'data center',
        'colocation', 'dedicated', 'alibaba', 'tencent', 'huawei'
    ]
    
    # 住宅ISP关键词
    RESIDENTIAL_KEYWORDS = [
        'telecom', 'mobile', 'broadband', 'communications', 'telephone',
        'cable', 'fiber', 'wireless', 'cellular', 'chinanet', 'unicom',
        'cmcc', 'verizon', 'at&t', 'comcast', 'spectrum', 'cox'
    ]
    
    async def test_connection(
        self,
        host: str,
        port: int,
        username: Optional[str] = None,
        password: Optional[str] = None,
        timeout: int = 15
    ) -> TestResult:
        """测试SOCKS5连接 - 使用底层socket测试认证"""
        import socket
        import struct
        
        start_time = time.time()
        
        try:
            # 1. 建立TCP连接
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.settimeout(timeout)
            sock.connect((host, port))
            
            # 2. SOCKS5握手
            if username and password:
                # 请求用户名密码认证
                sock.send(b'\x05\x02\x00\x02')
            else:
                # 请求无认证
                sock.send(b'\x05\x01\x00')
            
            response = sock.recv(2)
            
            if len(response) < 2:
                sock.close()
                return TestResult(success=False, message="代理响应无效")
            
            auth_method = response[1]
            
            # 3. 处理认证
            if auth_method == 0x02 and username and password:
                # 发送用户名密码
                auth_data = bytes([1, len(username)]) + username.encode() + bytes([len(password)]) + password.encode()
                sock.send(auth_data)
                auth_response = sock.recv(2)
                
                if len(auth_response) < 2 or auth_response[1] != 0x00:
                    sock.close()
                    return TestResult(success=False, message="认证失败：用户名或密码错误")
            elif auth_method == 0xFF:
                sock.close()
                return TestResult(success=False, message="代理拒绝连接：无可用认证方式")
            elif auth_method == 0x02 and not (username and password):
                sock.close()
                return TestResult(success=False, message="代理需要用户名密码认证")
            
            latency = int((time.time() - start_time) * 1000)
            
            # 4. 尝试通过代理连接测试目标
            # 使用多个测试目标，增加成功率
            test_targets = [
                ('www.google.com', 80),
                ('www.baidu.com', 80),
                ('httpbin.org', 80),
            ]
            
            region = None
            for target_host, target_port in test_targets:
                try:
                    # 发送连接请求
                    connect_request = b'\x05\x01\x00\x03' + bytes([len(target_host)]) + target_host.encode() + struct.pack('>H', target_port)
                    sock.send(connect_request)
                    connect_response = sock.recv(10)
                    
                    if len(connect_response) >= 2 and connect_response[1] == 0x00:
                        # 连接成功，尝试获取IP信息
                        sock.close()
                        
                        # 成功 - 认证通过且可以连接
                        return TestResult(
                            success=True,
                            message="连接成功（认证通过）",
                            latency=latency,
                            region=region
                        )
                except:
                    continue
            
            sock.close()
            
            # 认证成功但出口受限
            return TestResult(
                success=True,
                message="认证成功（出口可能受限）",
                latency=latency,
                region=None
            )
            
        except socket.timeout:
            return TestResult(success=False, message="连接超时")
        except ConnectionRefusedError:
            return TestResult(success=False, message="连接被拒绝：端口可能未开放")
        except Exception as e:
            return TestResult(success=False, message=f"测试失败: {str(e)}")
    
    async def check_ip_quality(
        self,
        host: str,
        port: int,
        username: Optional[str] = None,
        password: Optional[str] = None,
        timeout: int = 20
    ) -> IPCheckResult:
        """
        检测IP质量和TikTok适用性
        整合多个免费API来提升准确性
        """
        result = IPCheckResult(ip="")
        
        try:
            # 构建代理URL
            if username and password:
                proxy_url = f"socks5://{username}:{password}@{host}:{port}"
            else:
                proxy_url = f"socks5://{host}:{port}"
            
            # 创建代理连接器
            connector = ProxyConnector.from_url(proxy_url, rdns=True)
            
            async with aiohttp.ClientSession(connector=connector) as session:
                # 并行获取多个API的数据
                ip_api_data, ipinfo_data, ipapi_data = await asyncio.gather(
                    self._fetch_ip_api(session, timeout),
                    self._fetch_ipinfo(session, timeout),
                    self._fetch_ipapi_co(session, timeout),
                    return_exceptions=True
                )
                
                # 处理返回结果
                if isinstance(ip_api_data, Exception):
                    ip_api_data = None
                if isinstance(ipinfo_data, Exception):
                    ipinfo_data = None
                if isinstance(ipapi_data, Exception):
                    ipapi_data = None
                
                if not ip_api_data and not ipinfo_data and not ipapi_data:
                    result.tiktok_risk = "unknown"
                    result.risk_reasons.append("无法获取IP信息（代理可能限制出口）")
                    return result
                
                # 整合多源数据
                result = self._merge_ip_data(result, ip_api_data, ipinfo_data, ipapi_data)
                
                # 评估IP类型
                result = self._evaluate_ip_type(result)
                
                # 计算TikTok适用性评分
                result = self._calculate_tiktok_score(result)
                
                return result
                
        except Exception as e:
            result.tiktok_risk = "high"
            result.risk_reasons.append(f"检测失败: {str(e)}")
            return result
    
    async def _fetch_ip_api(self, session: aiohttp.ClientSession, timeout: int) -> Optional[Dict]:
        """ip-api.com - 基础IP信息"""
        try:
            async with session.get(
                "http://ip-api.com/json/?lang=zh-CN&fields=status,message,country,countryCode,region,regionName,city,isp,org,as,mobile,proxy,hosting,query",
                timeout=aiohttp.ClientTimeout(total=timeout)
            ) as response:
                if response.status == 200:
                    data = await response.json()
                    if data.get("status") == "success":
                        return data
        except Exception:
            pass
        return None
    
    async def _fetch_ipinfo(self, session: aiohttp.ClientSession, timeout: int) -> Optional[Dict]:
        """ipinfo.io - 补充IP类型信息（免费50k/月）"""
        try:
            async with session.get(
                "https://ipinfo.io/json",
                timeout=aiohttp.ClientTimeout(total=timeout)
            ) as response:
                if response.status == 200:
                    return await response.json()
        except Exception:
            pass
        return None
    
    async def _fetch_ipapi_co(self, session: aiohttp.ClientSession, timeout: int) -> Optional[Dict]:
        """ipapi.co - 补充ASN和安全信息（免费1000/天）"""
        try:
            async with session.get(
                "https://ipapi.co/json/",
                timeout=aiohttp.ClientTimeout(total=timeout)
            ) as response:
                if response.status == 200:
                    return await response.json()
        except Exception:
            pass
        return None
    
    def _merge_ip_data(self, result: IPCheckResult, ip_api: Optional[Dict], 
                       ipinfo: Optional[Dict], ipapi: Optional[Dict]) -> IPCheckResult:
        """整合多源数据，取最可靠的值"""
        
        # 优先使用 ip-api 的数据
        if ip_api:
            result.ip = ip_api.get("query", "")
            result.country = ip_api.get("country", "")
            result.country_code = ip_api.get("countryCode", "")
            result.region = ip_api.get("regionName", "")
            result.city = ip_api.get("city", "")
            result.isp = ip_api.get("isp", "")
            result.org = ip_api.get("org", "")
            result.as_name = ip_api.get("as", "")
            result.is_proxy = ip_api.get("proxy", False)
            result.is_datacenter = ip_api.get("hosting", False)
        
        # 用 ipinfo 补充
        if ipinfo:
            if not result.ip:
                result.ip = ipinfo.get("ip", "")
            if not result.country:
                result.country = ipinfo.get("country", "")
            if not result.city:
                result.city = ipinfo.get("city", "")
            if not result.org:
                result.org = ipinfo.get("org", "")
            
            # ipinfo 的 org 字段通常包含 ASN 信息
            org_lower = (ipinfo.get("org", "") or "").lower()
            if any(kw in org_lower for kw in self.DATACENTER_KEYWORDS):
                result.is_datacenter = True
        
        # 用 ipapi.co 补充
        if ipapi:
            if not result.ip:
                result.ip = ipapi.get("ip", "")
            if not result.country:
                result.country = ipapi.get("country_name", "")
            if not result.country_code:
                result.country_code = ipapi.get("country_code", "")
            if not result.city:
                result.city = ipapi.get("city", "")
            if not result.isp:
                result.isp = ipapi.get("org", "")
            if not result.as_name:
                result.as_name = ipapi.get("asn", "")
            
            # ipapi.co 提供的额外字段
            if ipapi.get("org"):
                org_lower = ipapi.get("org", "").lower()
                if any(kw in org_lower for kw in self.DATACENTER_KEYWORDS):
                    result.is_datacenter = True
        
        return result
    
    def _evaluate_ip_type(self, result: IPCheckResult) -> IPCheckResult:
        """评估IP类型"""
        isp_lower = (result.isp + " " + result.org + " " + result.as_name).lower()
        
        # 检查是否是数据中心IP
        for keyword in self.DATACENTER_KEYWORDS:
            if keyword in isp_lower:
                result.is_datacenter = True
                result.ip_type = "datacenter"
                break
        
        # 检查是否是住宅IP
        if not result.is_datacenter:
            for keyword in self.RESIDENTIAL_KEYWORDS:
                if keyword in isp_lower:
                    result.is_residential = True
                    result.ip_type = "residential"
                    break
        
        # 如果没有匹配到，根据其他标志判断
        if result.ip_type == "unknown":
            if result.is_proxy:
                result.ip_type = "proxy"
            elif result.is_datacenter:
                result.ip_type = "datacenter"
            else:
                result.ip_type = "residential"  # 默认假设是住宅
                result.is_residential = True
        
        return result
    
    def _calculate_tiktok_score(self, result: IPCheckResult) -> IPCheckResult:
        """
        计算TikTok适用性评分
        满分100分，基于多源数据综合评估
        
        核心逻辑：
        - 住宅IP是最重要的正面因素
        - 数据中心IP是最大的负面因素
        - "被检测为代理"对住宅IP影响小（因为住宅代理本身就是代理）
        """
        score = 100
        reasons = []
        positive_reasons = []
        
        # ============ 核心判断：IP类型 ============
        
        # 住宅IP是关键的正面因素
        is_residential_proxy = result.is_residential and not result.is_datacenter
        
        if result.is_datacenter:
            # 数据中心IP - 主要扣分项
            score -= 40
            reasons.append("数据中心/托管IP（-40分）")
            
            # 数据中心+被标记代理 = 额外扣分
            if result.is_proxy:
                score -= 10
                reasons.append("数据中心代理（额外-10分）")
        else:
            # 非数据中心IP
            if result.is_residential:
                # 住宅IP - 最好的情况
                score += 10
                positive_reasons.append("住宅IP（+10分）")
                
                # 住宅IP被标记为代理 - 轻微影响（住宅代理的正常情况）
                if result.is_proxy:
                    score -= 5
                    reasons.append("住宅代理（轻微影响-5分，属正常）")
            else:
                # 未知类型
                if result.is_proxy:
                    score -= 15
                    reasons.append("代理IP（-15分）")
        
        # ============ 地区评估 ============
        
        # TikTok主要市场（高权重）
        premium_countries = ['US', 'GB', 'JP', 'KR', 'TH', 'VN', 'ID', 'MY', 'PH', 'SG', 'TW', 'AU', 'CA', 'DE', 'FR', 'IT', 'ES', 'BR', 'MX', 'SA', 'AE']
        # TikTok受限/禁止地区
        restricted_countries = ['CN', 'IR', 'KP', 'SY', 'CU']
        
        if result.country_code in restricted_countries:
            score -= 50
            reasons.append(f"TikTok受限地区: {result.country}（-50分）")
        elif result.country_code in premium_countries:
            score += 5
            positive_reasons.append(f"TikTok主要市场: {result.country}（+5分）")
        
        # ============ ISP/ASN 分析 ============
        
        combined_info = f"{result.isp} {result.org} {result.as_name}".lower()
        
        # 云服务商IP（TikTok重点监控）
        cloud_providers = ['amazon', 'aws', 'google', 'gcp', 'microsoft', 'azure', 
                         'digitalocean', 'linode', 'vultr', 'alibaba', 'tencent', 'oracle cloud']
        for provider in cloud_providers:
            if provider in combined_info:
                score -= 15
                reasons.append(f"云服务商: {provider}（-15分）")
                break
        
        # 知名住宅运营商加分
        residential_isps = ['verizon', 'at&t', 'comcast', 'spectrum', 't-mobile', 'sprint',
                          'vodafone', 'telefonica', 'orange', 'bt group', 'virgin media',
                          'china telecom', 'china unicom', 'china mobile', 'chinanet',
                          'ntt', 'kddi', 'softbank', 'sk telecom', 'kt corporation',
                          'true internet', 'ais', 'dtac', 'globe telecom', 'pldt',
                          'telkomsel', 'indosat', 'xl axiata', 'maxis', 'celcom', 'digi']
        
        for isp in residential_isps:
            if isp in combined_info:
                score += 10
                positive_reasons.append(f"知名运营商（+10分）")
                break
        
        # ============ 最终计算 ============
        
        # 确保分数在0-100之间
        score = max(0, min(100, score))
        
        # 风险等级划分（调整阈值）
        if score >= 70:
            risk = "low"
        elif score >= 45:
            risk = "medium"
        else:
            risk = "high"
        
        # 整理显示信息
        all_reasons = []
        if positive_reasons:
            all_reasons.extend([f"✓ {r}" for r in positive_reasons])
        all_reasons.extend(reasons)
        
        result.tiktok_score = score
        result.tiktok_risk = risk
        result.risk_reasons = all_reasons
        
        return result
    
    async def get_ip_region(self, ip: str) -> Optional[str]:
        """获取IP归属地"""
        try:
            async with aiohttp.ClientSession() as session:
                async with session.get(
                    f"http://ip-api.com/json/{ip}?lang=zh-CN",
                    timeout=aiohttp.ClientTimeout(total=5)
                ) as response:
                    if response.status == 200:
                        data = await response.json()
                        if data.get("status") == "success":
                            return f"{data.get('country', '')} {data.get('city', '')}".strip()
        except Exception:
            pass
        
        return None
