"""
SSH服务：远程执行命令
"""
import asyncio
import time
from typing import Optional, Tuple
import asyncssh

from ..schemas.common import TestResult


class SSHService:
    """SSH服务"""
    
    async def test_connection(
        self, 
        host: str, 
        port: int, 
        username: str, 
        password: str,
        timeout: int = 10
    ) -> TestResult:
        """测试SSH连接"""
        start_time = time.time()
        
        try:
            async with asyncssh.connect(
                host=host,
                port=port,
                username=username,
                password=password,
                known_hosts=None,
                connect_timeout=timeout
            ) as conn:
                # 执行简单命令测试
                result = await conn.run('echo "ok"', timeout=5)
                
                latency = int((time.time() - start_time) * 1000)
                
                if result.exit_status == 0:
                    return TestResult(
                        success=True,
                        message="连接成功",
                        latency=latency
                    )
                else:
                    return TestResult(
                        success=False,
                        message=f"命令执行失败: {result.stderr}",
                        latency=latency
                    )
                    
        except asyncssh.PermissionDenied:
            return TestResult(
                success=False,
                message="认证失败：用户名或密码错误"
            )
        except asyncssh.ConnectionLost:
            return TestResult(
                success=False,
                message="连接丢失"
            )
        except asyncio.TimeoutError:
            return TestResult(
                success=False,
                message="连接超时"
            )
        except Exception as e:
            return TestResult(
                success=False,
                message=f"连接失败: {str(e)}"
            )
    
    async def execute_command(
        self,
        host: str,
        port: int,
        username: str,
        password: str,
        command: str,
        timeout: int = 60
    ) -> Tuple[bool, str, str]:
        """执行远程命令"""
        try:
            async with asyncssh.connect(
                host=host,
                port=port,
                username=username,
                password=password,
                known_hosts=None,
                connect_timeout=30
            ) as conn:
                result = await conn.run(command, timeout=timeout)
                
                return (
                    result.exit_status == 0,
                    result.stdout or "",
                    result.stderr or ""
                )
                
        except Exception as e:
            return False, "", str(e)
    
    async def execute_script(
        self,
        host: str,
        port: int,
        username: str,
        password: str,
        script_content: str,
        timeout: int = 300
    ) -> Tuple[bool, str, str]:
        """执行远程脚本"""
        try:
            async with asyncssh.connect(
                host=host,
                port=port,
                username=username,
                password=password,
                known_hosts=None,
                connect_timeout=30
            ) as conn:
                # 上传脚本
                script_path = "/tmp/deploy_script.sh"
                async with conn.start_sftp_client() as sftp:
                    async with sftp.open(script_path, 'w') as f:
                        await f.write(script_content)
                
                # 执行脚本
                result = await conn.run(f"chmod +x {script_path} && bash {script_path}", timeout=timeout)
                
                # 清理脚本
                await conn.run(f"rm -f {script_path}")
                
                return (
                    result.exit_status == 0,
                    result.stdout or "",
                    result.stderr or ""
                )
                
        except Exception as e:
            return False, "", str(e)
    
    async def upload_file(
        self,
        host: str,
        port: int,
        username: str,
        password: str,
        local_content: str,
        remote_path: str
    ) -> bool:
        """上传文件内容到远程服务器"""
        try:
            async with asyncssh.connect(
                host=host,
                port=port,
                username=username,
                password=password,
                known_hosts=None,
                connect_timeout=30
            ) as conn:
                async with conn.start_sftp_client() as sftp:
                    async with sftp.open(remote_path, 'w') as f:
                        await f.write(local_content)
                
                return True
                
        except Exception as e:
            return False




