"""
流量统计服务：从Xray获取流量数据
"""
import json
from datetime import datetime
from typing import Optional, Tuple, Dict, List
from sqlalchemy import select
from sqlalchemy.ext.asyncio import AsyncSession

from ..core.security import decrypt_string
from ..core.logger import get_logger
from ..models import Node, Server
from ..models.node import NodeStatus
from .ssh_service import SSHService

logger = get_logger(__name__)


class TrafficService:
    """流量统计服务"""
    
    def __init__(self, db: AsyncSession):
        self.db = db
        self.ssh_service = SSHService()
    
    async def get_traffic_from_server(
        self,
        server: Server,
        node_id: int
    ) -> Tuple[bool, int, int]:
        """
        从服务器获取节点流量统计
        返回: (success, upload_bytes, download_bytes)
        """
        server_password = decrypt_string(server.ssh_password)
        if not server_password:
            server_password = server.ssh_password
        
        # 获取节点对应的服务名和配置
        service_name = f"xray-node-{node_id}"
        
        # 更健壮的流量统计脚本 - 使用模板替换避免f-string转义问题
        script_template = r'''#!/bin/bash
# 获取Xray流量统计 - 节点 NODE_ID_PLACEHOLDER

# 找到Xray路径
XRAY_BIN=""
for path in /usr/local/bin/xray /usr/bin/xray /opt/xray/xray $(which xray 2>/dev/null); do
    if [ -x "$path" ]; then
        XRAY_BIN="$path"
        break
    fi
done

if [ -z "$XRAY_BIN" ]; then
    echo "0 0 ERROR:xray_not_found"
    exit 0
fi

# 检查服务是否运行
SERVICE_NAME_PLACEHOLDER="SERVICE_PLACEHOLDER"
if ! systemctl is-active --quiet "$SERVICE_NAME_PLACEHOLDER"; then
    echo "0 0 ERROR:service_not_running"
    exit 0
fi

# Xray API地址
API_SERVER="127.0.0.1:10085"

# 检查API端口是否监听（多种方法尝试）
API_LISTENING=false
if ss -tlnp 2>/dev/null | grep -q ":10085"; then
    API_LISTENING=true
elif netstat -tlnp 2>/dev/null | grep -q ":10085"; then
    API_LISTENING=true
elif lsof -i:10085 >/dev/null 2>&1; then
    API_LISTENING=true
fi

if [ "$API_LISTENING" = "false" ]; then
    echo "0 0 ERROR:api_not_listening"
    exit 0
fi

# 获取流量统计 - 尝试多种方式
get_traffic() {
    local stat_name="$1"
    local result=""
    local value=""
    
    # 方法1: 使用 xray api stats 命令
    result=$("$XRAY_BIN" api stats --server=$API_SERVER -name "$stat_name" 2>/dev/null)
    
    if [ -n "$result" ]; then
        # 尝试多种解析方式
        # 方式1: 使用awk解析JSON
        value=$(echo "$result" | awk -F'"' '/"value"/{gsub(/[^0-9]/,"",$4); print $4}' | head -1)
        
        # 方式2: 如果awk失败，用grep和sed
        if [ -z "$value" ] || ! [[ "$value" =~ ^[0-9]+$ ]]; then
            value=$(echo "$result" | grep -o '"value"[[:space:]]*:[[:space:]]*"[0-9]*"' | grep -o '[0-9]*' | head -1)
        fi
        
        # 方式3: 尝试直接数字匹配
        if [ -z "$value" ] || ! [[ "$value" =~ ^[0-9]+$ ]]; then
            value=$(echo "$result" | grep -oE '[0-9]+' | tail -1)
        fi
        
        if [ -n "$value" ] && [[ "$value" =~ ^[0-9]+$ ]]; then
            echo "$value"
            return
        fi
    fi
    
    echo "0"
}

# 获取入站流量统计
UP=$(get_traffic "inbound>>>vless-in>>>traffic>>>uplink")
DOWN=$(get_traffic "inbound>>>vless-in>>>traffic>>>downlink")

# 确保是数字
if ! [[ "$UP" =~ ^[0-9]+$ ]]; then UP=0; fi
if ! [[ "$DOWN" =~ ^[0-9]+$ ]]; then DOWN=0; fi

echo "$UP $DOWN OK"
'''
        # 替换占位符
        command = script_template.replace("NODE_ID_PLACEHOLDER", str(node_id))
        command = command.replace("SERVICE_PLACEHOLDER", service_name)
        
        success, stdout, stderr = await self.ssh_service.execute_script(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=server_password,
            script_content=command,
            timeout=30
        )
        
        logger.debug(f"节点 {node_id} 流量查询结果: success={success}, stdout='{stdout}', stderr='{stderr}'")
        
        if stdout:
            try:
                parts = stdout.strip().split()
                if len(parts) >= 2:
                    up = int(parts[0]) if parts[0].isdigit() else 0
                    down = int(parts[1]) if parts[1].isdigit() else 0
                    
                    # 检查是否有错误信息
                    if len(parts) >= 3 and parts[2].startswith("ERROR:"):
                        error_type = parts[2].replace("ERROR:", "")
                        logger.warning(f"节点 {node_id} 流量获取问题: {error_type}")
                    else:
                        logger.info(f"节点 {node_id} 流量: 上传={up}, 下载={down}")
                    
                    return True, up, down
            except (ValueError, IndexError) as e:
                logger.error(f"节点 {node_id} 解析流量数据失败: {e}, stdout='{stdout}'")
        else:
            logger.warning(f"节点 {node_id} 流量查询无输出, stderr='{stderr}'")
        
        return False, 0, 0
    
    async def sync_node_traffic(self, node: Node, server: Server) -> bool:
        """同步单个节点的流量"""
        try:
            success, up, down = await self.get_traffic_from_server(server, node.id)
            
            if success:
                node.traffic_up = up
                node.traffic_down = down
                node.traffic_total = up + down
                node.last_traffic_sync = datetime.now()
                
                # 检查是否超出流量限制
                if node.traffic_limit and node.traffic_limit > 0:
                    if node.traffic_total >= node.traffic_limit:
                        node.is_traffic_exceeded = True
                else:
                    node.is_traffic_exceeded = False
                
                await self.db.commit()
                return True
        except Exception as e:
            print(f"同步节点 {node.id} 流量失败: {e}")
        
        return False
    
    async def sync_all_nodes_traffic(self) -> Dict[str, int]:
        """同步所有运行中节点的流量"""
        result = {"success": 0, "failed": 0, "skipped": 0}
        
        # 获取所有运行中的节点
        query = select(Node).where(Node.status == NodeStatus.RUNNING)
        nodes_result = await self.db.execute(query)
        nodes = nodes_result.scalars().all()
        
        for node in nodes:
            server = await self.db.get(Server, node.server_id)
            if not server:
                result["skipped"] += 1
                continue
            
            success = await self.sync_node_traffic(node, server)
            if success:
                result["success"] += 1
            else:
                result["failed"] += 1
        
        return result
    
    async def check_node_expiry(self, node: Node) -> bool:
        """检查节点是否过期"""
        if node.expire_at:
            now = datetime.now()
            if now >= node.expire_at:
                node.is_expired = True
                await self.db.commit()
                return True
            else:
                node.is_expired = False
        return False
    
    async def check_and_stop_exceeded_nodes(self) -> List[int]:
        """检查并停止流量超限或过期的节点"""
        stopped_nodes = []
        
        # 获取所有运行中的节点
        query = select(Node).where(Node.status == NodeStatus.RUNNING)
        nodes_result = await self.db.execute(query)
        nodes = nodes_result.scalars().all()
        
        for node in nodes:
            should_stop = False
            reason = ""
            
            # 检查到期
            if node.expire_at:
                if datetime.now() >= node.expire_at:
                    node.is_expired = True
                    should_stop = True
                    reason = "套餐已过期"
            
            # 检查流量超限
            if node.traffic_limit and node.traffic_limit > 0:
                if node.traffic_total >= node.traffic_limit:
                    node.is_traffic_exceeded = True
                    should_stop = True
                    reason = "流量已超限"
            
            if should_stop:
                # 停止节点
                server = await self.db.get(Server, node.server_id)
                if server:
                    try:
                        await self._stop_node_on_server(server, node)
                        node.status = NodeStatus.STOPPED
                        stopped_nodes.append(node.id)
                        print(f"节点 {node.id} ({node.name}) 已自动停止: {reason}")
                    except Exception as e:
                        print(f"自动停止节点 {node.id} 失败: {e}")
        
        await self.db.commit()
        return stopped_nodes
    
    async def _stop_node_on_server(self, server: Server, node: Node):
        """在服务器上停止节点"""
        server_password = decrypt_string(server.ssh_password)
        service_name = f"xray-node-{node.id}"
        
        command = f"systemctl stop {service_name}"
        
        success, stdout, stderr = await self.ssh_service.execute_command(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=server_password,
            command=command,
            timeout=30
        )
        
        if not success:
            raise Exception(f"停止服务失败: {stderr}")
    
    async def reset_node_traffic(self, node: Node, server: Server) -> bool:
        """重置节点流量统计"""
        server_password = decrypt_string(server.ssh_password)
        
        # 重置Xray流量统计
        command = '''#!/bin/bash
# 尝试重置Xray流量统计
/usr/local/bin/xray api stats --server=127.0.0.1:10085 -reset 2>/dev/null || true
echo "ok"
'''
        
        success, stdout, stderr = await self.ssh_service.execute_script(
            host=server.ip,
            port=server.ssh_port,
            username=server.ssh_user,
            password=server_password,
            script_content=command,
            timeout=30
        )
        
        # 无论远程重置是否成功，都重置本地记录
        node.traffic_up = 0
        node.traffic_down = 0
        node.traffic_total = 0
        node.is_traffic_exceeded = False
        node.last_traffic_sync = datetime.now()
        
        await self.db.commit()
        return True
    
    def calculate_days_remaining(self, expire_at: Optional[datetime]) -> Optional[int]:
        """计算剩余天数"""
        if not expire_at:
            return None
        
        now = datetime.now()
        if now >= expire_at:
            return 0
        
        delta = expire_at - now
        return delta.days
    
    def calculate_traffic_percent(self, used: int, limit: Optional[int]) -> float:
        """计算流量使用百分比"""
        if not limit or limit <= 0:
            return 0.0
        
        return min(100.0, (used / limit) * 100)
    
    def format_bytes(self, bytes_val: int) -> str:
        """格式化字节数为可读字符串"""
        if bytes_val < 1024:
            return f"{bytes_val} B"
        elif bytes_val < 1024 * 1024:
            return f"{bytes_val / 1024:.2f} KB"
        elif bytes_val < 1024 * 1024 * 1024:
            return f"{bytes_val / (1024 * 1024):.2f} MB"
        else:
            return f"{bytes_val / (1024 * 1024 * 1024):.2f} GB"
