"""
全球节点自动化搭建工具 - 服务器模块 (打包版)
"""
import os
import sys
from pathlib import Path
from contextlib import asynccontextmanager

# 获取基础目录
if getattr(sys, 'frozen', False):
    BASE_DIR = Path(sys.executable).parent
    # PyInstaller打包后，静态文件在_internal目录
    INTERNAL_DIR = BASE_DIR / "_internal"
else:
    BASE_DIR = Path(__file__).parent
    INTERNAL_DIR = BASE_DIR

# 设置路径
sys.path.insert(0, str(BASE_DIR))

from fastapi import FastAPI, Request
from fastapi.middleware.cors import CORSMiddleware
from fastapi.staticfiles import StaticFiles
from fastapi.responses import FileResponse, HTMLResponse

def create_app():
    """创建FastAPI应用"""
    
    # 导入必要模块
    from app.core.database import init_db
    from app.api import api_router
    
    @asynccontextmanager
    async def lifespan(app: FastAPI):
        # 启动时初始化数据库
        await init_db()
        print("[*] 数据库初始化完成")
        yield
    
    app = FastAPI(
        title="全球节点自动化搭建工具",
        version="2.3.1",
        lifespan=lifespan
    )
    
    # CORS配置
    app.add_middleware(
        CORSMiddleware,
        allow_origins=["*"],
        allow_credentials=True,
        allow_methods=["*"],
        allow_headers=["*"],
    )
    
    # 注册API路由（必须在静态文件之前）
    app.include_router(api_router, prefix="/api")
    
    # 静态文件目录 - 检查多个可能的位置
    static_dir = None
    for possible_dir in [INTERNAL_DIR / "static", BASE_DIR / "static", BASE_DIR / "_internal" / "static"]:
        if possible_dir.exists():
            static_dir = possible_dir
            print(f"[*] 找到静态文件目录: {{static_dir}}")
            break
    
    if static_dir and static_dir.exists():
        index_file = static_dir / "index.html"
        assets_dir = static_dir / "assets"
        images_dir = static_dir / "images"
        
        # 挂载assets目录
        if assets_dir.exists():
            app.mount("/assets", StaticFiles(directory=str(assets_dir)), name="assets")
            print(f"[*] 挂载assets目录: {{assets_dir}}")
        
        # 挂载images目录
        if images_dir.exists():
            app.mount("/images", StaticFiles(directory=str(images_dir)), name="images")
            print(f"[*] 挂载images目录: {{images_dir}}")
        
        # 读取index.html内容
        if index_file.exists():
            index_content = index_file.read_text(encoding="utf-8")
            
            @app.get("/", response_class=HTMLResponse)
            async def serve_index():
                return HTMLResponse(content=index_content)
            
            @app.get("/favicon.svg")
            async def serve_favicon():
                favicon_path = static_dir / "favicon.svg"
                if favicon_path.exists():
                    return FileResponse(str(favicon_path), media_type="image/svg+xml")
                return HTMLResponse(content="", status_code=404)
            
            @app.get("/{{full_path:path}}", response_class=HTMLResponse)
            async def serve_spa(request: Request, full_path: str):
                # API路径不处理
                if full_path.startswith("api"):
                    return HTMLResponse(content='{{"detail":"Not Found"}}', status_code=404)
                
                # 跳过已挂载的目录
                if full_path.startswith("assets") or full_path.startswith("images"):
                    return HTMLResponse(content='{{"detail":"Not Found"}}', status_code=404)
                
                # 尝试返回静态文件
                file_path = static_dir / full_path
                if file_path.exists() and file_path.is_file():
                    return FileResponse(str(file_path))
                
                # 其他路径返回index.html（SPA路由）
                return HTMLResponse(content=index_content)
    else:
        print("[!] 警告: 未找到静态文件目录")
    
    return app
